const connectBtn = document.getElementById('connect-btn');
const disconnectBtn = document.getElementById('disconnect-btn');
const statusElement = document.getElementById('status');
const dashboard = document.getElementById('dashboard');

// Sets up the event listeners and initialize the UI
function init() {
    connectBtn.addEventListener('click', onConnectClick);
    disconnectBtn.addEventListener('click', onDisconnectClick);
    
    document.addEventListener('ws-connected', onWebSocketConnected);
    document.addEventListener('ws-message', onWebSocketMessage);
    document.addEventListener('ws-disconnected', onWebSocketDisconnected);
}

// Connect button click handler
function onConnectClick() {
    statusElement.textContent = 'Connecting...';
    dashboard.innerHTML = '<div class="message">Connecting to WebSocket...</div>';
    connectBtn.disabled = true;
    
    // Call WebSocket connect function from websocket.js
    connectToWebSocket();
}

// Disconnect button click handler
function onDisconnectClick() {
    // Call WebSocket disconnect function from websocket.js
    disconnectFromWebSocket();
}

// WebSocket connected event handler
function onWebSocketConnected() {
    // Update UI
    statusElement.textContent = 'Connected';
    statusElement.className = 'status connected';
    connectBtn.disabled = true;
    disconnectBtn.disabled = false;
    
    // Create crypto cards
    dashboard.innerHTML = '';
    getCryptoSymbols().forEach(createCryptoCard);
}

// WebSocket message event handler
function onWebSocketMessage(event) {
    // Get data from event
    const data = event.detail;
    updateCryptoCard(data);
}

// WebSocket disconnected event handler
function onWebSocketDisconnected() {
    // Update UI
    statusElement.textContent = 'Disconnected';
    statusElement.className = 'status disconnected';
    connectBtn.disabled = false;
    disconnectBtn.disabled = true;
    dashboard.innerHTML = '<div class="message">WebSocket disconnected</div>';
}

// Create a card for a cryptocurrency
function createCryptoCard(symbol) {
    const card = document.createElement('div');
    card.className = 'crypto-card';
    card.id = `card-${symbol}`;
    
    card.innerHTML = `
        <div class="symbol">${symbol.toUpperCase()}</div>
        <div class="price" id="price-${symbol}">Loading...</div>
        <div class="change" id="change-${symbol}">0.00%</div>
    `;
    
    dashboard.appendChild(card);
}

// Update crypto card with new data
function updateCryptoCard(data) {
    const symbol = data.s.toLowerCase();
    const price = parseFloat(data.c);
    const change = parseFloat(data.P);
    
    const priceElement = document.getElementById(`price-${symbol}`);
    const changeElement = document.getElementById(`change-${symbol}`);
    
    if (priceElement && changeElement) {
        // Update price and change values
        priceElement.textContent = '$' + price.toFixed(2);
        changeElement.textContent = change.toFixed(2) + '%';
        
        // Update styling based on price change
        priceElement.className = 'price ' + (change > 0 ? 'price-up' : 'price-down');
        changeElement.className = 'change ' + (change > 0 ? 'positive' : 'negative');
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', init);