/*

Example implementation of generic binary search tree.

*/


type NullOr<T> = null | T;

class myNode<T> {
  data: T;

  leftChild: NullOr<myNode<T>>;
  rightChild: NullOr<myNode<T>>;

  constructor(data: T) {
    this.data = data;  // This may not be the best way to do it. Maybe cloning would be a better way.
    this.leftChild = null;
    this.rightChild = null;
  }

  isLeaf(): boolean {
    return this.leftChild === null && this.rightChild === null;
  }
}

/*
  Binary search tree.
  - 'le()' is the predicate "less than or equal to" and it gives linear ordering of the values of the
    type parameter (Note: Linear ordering might not be necessary for the BST to function properly.
    You can break this rule at your own risk.).
  - can contain only unique elements according to the provided 'le()' predicate from the type parameter.
  - all left children of a node are less than or equal to it.
  - all right children of a node are strictly greater than it.
*/
class BST<T extends { le(t: T): boolean }> {
  
  private root: myNode<T>;

  // This is the only constructor and it accepts data for the root node, so the BST cannot be empty.
  constructor(data: T) {
    this.root = new myNode(data);
  }

  // Add node with data 'data' in the BST.
  add(data: T): boolean {
    return this.addFromNode(data);
  }

  // Returns 'null' if 'data' was not found in the BST.
  find(data: T): NullOr<T> {
    return this.findFromNode(data);
  }

  // Returns an array of arrays.
  // The i-th element of the returned array will be an array of all data at i-th level of the tree.
  // Data in each array is ordered from left to right.
  returnByLevels(): T[][] {
    const levels: T[][] = [];
    this.returnByLevelsFromNode(levels);
    return levels;
  }

  // Returns true if 't1' and 't2' are equal.
  private eq(t1: T, t2: T): boolean {
    return t1.le(t2) && t2.le(t1);
  }

  // Add node with data 'data' to the BST with root 'node'.
  // If 'data' is already in the tree it WON'T be added and false will be returned.
  private addFromNode(data: T, node: myNode<T> = this.root): boolean {
    if (data.le(node.data)) {
      if (node.data.le(data))  // Check if data is equal
        return false;          // and don't add it to the BST if it is.
      
      if (node.leftChild === null) {
        node.leftChild = new myNode<T>(data);
        return true;
      }
      
      return this.addFromNode(data, node.leftChild);
    }
    else {
      if (node.rightChild === null) {
        node.rightChild = new myNode<T>(data);
        return true;
      }
      
      return this.addFromNode(data, node.rightChild);
    }
  }

  // Searches in the BST with root 'node' and returns if a node has equal data to 'data'.
  // Returns 'null' if no match was found.
  private findFromNode(data: T, node: NullOr<myNode<T>> = this.root): NullOr<T> {
    if (node === null)
      return null;      // No match found.
    
    if (this.eq(data, node.data))
      return node.data;
    
    if (data.le(node.data)) 
      return this.findFromNode(data, node.leftChild);
    else
      return this.findFromNode(data, node.rightChild);
  }

  // Writes the nodes level by level in the 'levels' array.
  // When done i-th element of 'levels' will be array of all data at i-th level.
  // Order of nodes is from left to right.
  private returnByLevelsFromNode
  (levels: T[][], i: number = 0, node: NullOr<myNode<T>> = this.root): void {
    if (node === null) return;

    if (levels[i] === undefined)
      levels.push([]);           // This is the first time we reached this level.

    levels[i].push(node.data);

    this.returnByLevelsFromNode(levels, i + 1, node.leftChild);
    this.returnByLevelsFromNode(levels, i + 1, node.rightChild);
  }
}

// For testing out 'BST<T>' class.
class Num {
  n: number;
  
  constructor(num: number) { this.n = num; }

  le(num: Num): boolean { return this.n <= num.n; }
}

// Test 'BST<T>'.
// Note: I don't know how to write proper unit tests yet, don't judge me :(
{
  function createBST(): BST<Num> {
    /*
           0
          / \
        -4   3
        / \   \
      -9  -2   6
          /     \
        -3       8
    */
    const bst = new BST(new Num(0));
    
    bst.add(new Num(-4));    bst.add(new Num(3));
    bst.add(new Num(-9));    bst.add(new Num(-2));    bst.add(new Num(6));
    bst.add(new Num(-3));    bst.add(new Num(8));

    return bst;  
  }
  
  // Prints to console if the test succeed or not.
  function testAdd(): void {

    function failure() { isSuccessful = false; console.log("Failed 'testAdd()'"); }

    let isSuccessful = true;

    const bst = createBST();
    
    if (!bst.add(new Num(-8))) failure();
    if (bst.find(new Num(-8)) === null) failure();

    if (!bst.add(new Num(7))) failure();
    if (bst.find(new Num(7)) === null) failure();

    // Elements must be unique.
    if (bst.add(new Num(-3))) failure();

    if (isSuccessful) console.log("Passed 'testAdd()'");
  }

  // Prints to console if the test succeed or not.
  function testFind(): void {

    function failure() { isSuccessful = false; console.log("Failed 'testFind()'"); }

    let isSuccessful = true;

    const bst = createBST();

    if (bst.find(new Num(42)) !== null) failure();

    let res = bst.find(new Num(0));
    if (res === null || res.n !== 0) failure();

    res = bst.find(new Num(-3));
    if (res === null || res.n !== -3) failure();

    if (isSuccessful) console.log("Passed 'testFind()'");
  }

  // Prints to console if the test succeed or not.
  function testReturnByLevels(): void {

    function failure() { isSuccessful = false; console.log('Failed testReturnByLevels().'); }

    let isSuccessful = true;

    const bst = createBST();

    const res = bst.returnByLevels();

    const expected = [ [ 0 ], [ -4, 3 ], [ -9, -2, 6 ], [ -3, 8 ] ];
    for (let i = 0; i < expected.length; ++i)
      for (let j = 0; j < expected[i].length; ++j)
        if (res[i][j].n !== expected[i][j]) {
          failure();
          return;
        }
    
    if (isSuccessful) console.log("Passed 'testReturnByLevels()'");
  }

  testAdd();
  testFind();
  testReturnByLevels();
}