// "use strict";

/**
 * This is a special keyword in JavaScript
 * that refers to the current execution context.
 */

/**
 * Global context
 * In a global scope, `this` refers to the global object.
 * In a browser, `this` is the window object.
 * In Node.js, `this` is the global object.
 */
// console.log(this === global); // true in Node.js
console.log(this === window); // true in a browser

/**
 * Function context
 * In a non-method function call (non strict mode),
 * `this` refers to the global/window object
 */
function myFunction() {
  // console.log(this === global); // true in Node.js
  console.log(this === window); // true in browser
  console.log(this); // global/window object or undefined in strict mode
}
myFunction();

/**
 * Method context
 * When a function is called as a method,
 * this is set to the object the method is called on.
 */
function sayHello() {
  console.log("Hello, " + this.name); // 'Hello, John'
  console.log(this); // { name: 'John', sayHello: [Function: sayHello] }
}

const obj = {
  name: "John",
  sayHello,
};
obj.sayHello();

/**
 * Constructor function
 * When a function is called with the new keyword,
 * `this` is set to the newly created object.
 */
function Car(make, model, year) {
  this.make = make;
  this.model = model;
  this.year = year;
}

const myCar = new Car("Toyota", "Corolla", 2005);
console.log(myCar.model); // 'Corolla'

/**
 * `call`, `apply` and `bind` context
 * The methods call, apply, bind allow you to
 * decide the value of `this`.
 */
const cat = {
  sound: "meow",
};

const dog = {
  sound: "woof",
};

function makeSound() {
  console.log(this.sound);
}

function makeMultipleSounds(n, extraSound) {
  while (n > 0) {
    console.log(this.sound + " " + extraSound);
    n--;
  }
}

makeSound(); // undefined
makeMultipleSounds(3, "extra"); // 'undefined extra' 'undefined extra' 'undefined extra'

/**
 * `apply` and `call` work the same,
 * but `apply` takes an array-like object of arguments,
 * while `call` takes an argument list.
 */
makeSound.call(cat); // 'meow'
makeMultipleSounds.call(dog, 3, "call"); // 'woof call' 'woof call' 'woof call'

makeSound.apply(cat); // 'meow'
makeMultipleSounds.apply(cat, [3, "apply"]); // 'meow apply' 'meow apply' 'meow apply'

/**
 * `bind` returns a new function, allowing you
 * to pass in a `this` of any object
 */
const catSound = makeSound.bind(cat);
catSound(); // 'meow'

const dogSound = makeSound.bind(dog);
dogSound(); // 'woof'

/**
 * Arrow function context
 * `this` in an arrow function behaves differently.
 * It captures the `this` value of the enclosing context. (no binding)
 */
const objArrow = {
  name: "John",
  sayHello: () => {
    console.log("Hello, " + this.name); // 'Hello, '
  },
  sayHello2: function () {
    console.log("Hello2, " + this.name); // 'Hello2, John'
  }, // same as the function sayHello in the first example
  sayHello3: function () {
    setTimeout(() => {
      console.log("Hello3, " + this.name); // 'Hello3, John'
    }, 1);
  },
};
objArrow.sayHello();
objArrow.sayHello2();
objArrow.sayHello3();
