/* 
============================================================
Презентация: Closures в JavaScript
============================================================

Основни бележки и теоретични акценти:
------------------------------------------------------------
- Областта на видимост в една функция "наследява" обкръжаващата я област (closure).
- "Вложената" функция може да променя стойностите в обкръжението си.
- Променливите в JavaScript могат да принадлежат към:
    • Локална област на видимост
    • Глобална област на видимост
- Глобалните променливи могат да бъдат направени локални (частни) чрез closures.
- Closures позволяват на функцията да има "частни" променливи.
- В уеб страница глобалните променливи принадлежат към самата страница и могат да бъдат достъпни или променяни от всички скриптове.
- Локална променлива може да бъде използвана само вътре във функцията, в която е дефинирана – тя е частна и скрита от други функции и код.
- Глобални и локални променливи със същото име са различни – промяната на едната не засяга другата.
- Локалните променливи имат кратък живот: създават се при извикване на функцията и се унищожават, когато тя приключи.
- Лексикалната среда (Lexical Environment) в JavaScript е структура от данни, която съхранява променливите и функциите, дефинирани в текущата област на видимост, заедно с референции към всички външни области на видимост. Известна е още като лексикална област на видимост (lexical scope).
- Лексикалната среда се създава в момента, в който една функция бъде дефинирана, и СЕ ЗАПАЗВА, докато функцията или някое от нейните closures остава достъпно.
- Можем да си мислим за "closure" като „функция с памет“ – тя запазва стойностите на променливите между извикванията (например, променливата `count` не се нулира, защото closure „помни“ нейното предишно състояние).
============================================================
*/


/* ============================================================
Слайд 1: Функции като първокласни граждани – пътят към Closures
------------------------------------------------------------
Текст: Какво става с променливите, които са били част от една функция, но трябва да се използват по-късно?
Пример:
============================================================ */
function outer() {
    let message = "Здравей";
    return function inner() {
        console.log(message);
    };
}

const sayHello = outer();
sayHello(); // Изход: "Здравей"


/* ============================================================
Слайд 2: Какво представляват Closures?
------------------------------------------------------------
1) Closure е функция, която запазва достъп до обкръжаващата я лексикална област, дори след като тази област е затворена.
2) Closure позволява на вътрешната функция да „помни“ променливите от обкръжаващата я функция, дори след като тя е изпълнена.
------------------------------------------------------------
============================================================ */


/* ============================================================
Слайд 3: Пример 01 – Брояч (Counter)
Част 1: Глобална променлива – лоша практика
------------------------------------------------------------
Проблем: Използването на глобална променлива за брояч води до нежелани промени.
============================================================ */
let counter = 0;

function incrementGlobal() {
    ++counter;
}

incrementGlobal();
incrementGlobal();
incrementGlobal();
console.log(counter); // Изход: 3

// Променяме стойността директно:
counter = 100000;
console.log(counter); // Изход: 100000

// Въпрос: Как можем да "скрием" и защитим стойността на брояча?


/* ============================================================
Слайд 4: Пример 01 – Брояч (Counter)
Част 2: Скриване на брояча, но без запазване на състоянието
------------------------------------------------------------
Опитваме се да скрием брояча, като го дефинираме вътре във функцията, но стойността му винаги се рестартира.
============================================================ */
function incrementLocal() {
    let counter = 0;
    return ++counter;
}

console.log(incrementLocal()); // Изход: 1
console.log(incrementLocal()); // Изход: 1
console.log(incrementLocal()); // Изход: 1

// Въпрос: Как да запазим стойността на брояча между извикванията?


/* ============================================================
Слайд 5: Пример 01 – Брояч (Counter)
Част 3: Магията на Closure – запазване на състоянието
------------------------------------------------------------
Създаваме функция, която връща вътрешна функция, запазвайки променливата "count" в closure.
============================================================ */
function createCounter() {
    let count = 0; // Броячът е скрит в closure
    return function () {
        return ++count;
    };
}

const counterClosure = createCounter(); // Създаваме брояч само веднъж
console.log(counterClosure()); // Изход: 1
console.log(counterClosure()); // Изход: 2
console.log(counterClosure()); // Изход: 3


/* ============================================================
Слайд 6: Пример 02 – Бутони за промяна на размера на текста
Част 1: Пример без използване на closure
------------------------------------------------------------
Използваме глобална променлива за размера на шрифта.
============================================================ */

/*
let size = 12;

document.getElementById("size-btn").onclick = function () {
    document.body.style.fontSize = `${size}px`;
};

// Въпрос: Как да създаваме динамично различни размери за текста?
*/


/* ============================================================
Слайд 7: Пример 02 – Бутони за промяна на размера на текста
Част 2: Използване на closure за задаване на специфичен размер
------------------------------------------------------------
Функцията makeSizer връща функция, която използва зададения размер за шрифта.
============================================================ */

/*
function makeSizer(size) {
    return function () {
        document.body.style.fontSize = `${size}px`;
    };
}

const size12 = makeSizer(12);
const size14 = makeSizer(14);
const size16 = makeSizer(16);

document.getElementById("size-12").onclick = size12;
document.getElementById("size-14").onclick = size14;
document.getElementById("size-16").onclick = size16;

// Всяка функция "помни" своя размер и го използва при клик.
*/

/* ============================================================
Слайд 8: Пример 03 – Проследяване на резултат в играта (Game Score Tracker)
Част 1: Инициализация на играта с проследяване на резултата
------------------------------------------------------------
Дефинираме функции за увеличаване и намаляване на резултата, както и за извличането му.
============================================================ */
function initGame() {
    let score = 0;

    function increaseScore(points) {
        score += points;
        console.log(`+${points} точки. Браво!`);
    }

    function decreaseScore(points) {
        score -= points;
        console.log(`-${points} точки. Можеш и по-добре!`);
    }

    function getScore() {
        return score;
    }

    return {
        increaseScore,
        decreaseScore,
        getScore
    };
}


/* ============================================================
Слайд 9: Пример 03 – Проследяване на резултат в играта (Game Score Tracker)
Част 2: Тестване на играта
------------------------------------------------------------
Използваме initGame, за да проследим резултата в два рунда на играта.
============================================================ */
const game = initGame();

console.log("Нека играта започне!");

// Първи рунд
game.increaseScore(3);
game.decreaseScore(2);
console.log(`
  Рунд 1 приключи! 
  Текущият ти резултат е ${game.getScore()} точки. 
  Готов ли си за Рунд 2... Напред!
  `);

// Втори рунд
game.decreaseScore(2);
game.increaseScore(6);
console.log(`
  Играта приключи. 
  Крайният ти резултат е ${game.getScore()} точки! 
  Благодарим ти, че игра!
  `);


/* ============================================================
Слайд 10: Изход от конзолата
------------------------------------------------------------
Тук се показва примерният изход от конзолата за проследяване на резултат.
============================================================ */


/* ============================================================
Слайд 11: Финален слайд
------------------------------------------------------------
(Включи подходящ gif или meme, за да завършиш презентацията с усмивка!)
============================================================ */