#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>


// Връща произволно число в целия диапазон на int (0, MAX_INT), равномерно разпределено.
// rand() връща число то 0 до MAX_RAND, което е гарантирано да бъде поне 32XXX
// Твърде малко за тестовете.
int random() {
	int random =  rand() & 255;
	random = random<<8 | (rand() & 255);
	random = random<<8 | (rand() & 255);
	random = random<<7 | (rand() & 127);

	return random;
}

/*
	-------------------------------------------------------------------------
	Хеширащи функции за int. За реализацията на алгоритъма ни трявват няколко - а.
	Използвам 4. Някои не са много добри. За реални приложения би трябвало да се
	ползват някакви хеширащи алгоритми. Там най-вероятна ще се хешират низове,
	за които саществуват такива.
	-------------------------------------------------------------------------
*/

typedef unsigned int Hash;
typedef Hash (*HashFunction)(int);

Hash hash1(int v) { return (Hash)v; }
Hash hash2(int v) { return (Hash)(~v); }
Hash hash3(int v) { return (Hash)(179426549 * v); }
Hash hash4(int v) { return (Hash)(~(179426549 * v)); }

// брой хеш функции
#define D 4
// броят различни елементи в поткар 2 на степен 20
#define N 1048576
#define LOGN 20
// размер на буфера на всяка скица
// очакваме f + 2M/W да е горната граница на оценката за f,
// където М е размерът на потока
#define W 1000

typedef unsigned int Count;
typedef struct Sketch {
	HashFunction hasher;
	Count cs[W];
} Sketch;

// запазва хеш функцията и нулира всички броячи в скицата
void initSketch(Sketch* sketch, HashFunction h) {
	sketch->hasher = h;
	memset(sketch->cs, 0, W * sizeof (Count));
}

// добява нова стойност
Count getCountSketch(Sketch* sketch, int v) {
	Hash h = sketch->hasher(v) % W;
	return sketch->cs[h];
}

// връща оценка за броя срещания на числото дотук
void addSkech(Sketch* sketch, int v) {
	Hash h = sketch->hasher(v) % W;
	++sketch->cs[h];
}

/*
	-------------------------------------------------------------------------
	Реализация на Count Min Sketch, която използва D на брой Sketch-a.
	-------------------------------------------------------------------------
*/

#define MIN(x, y) (((x) < (y)) ? (x) : (y))

typedef struct CMSketch {
	Sketch ss[D];
} CMSketch;


// инициализра четерите Sketch-a с съответните хеш функции
void initCMSketch(CMSketch* cms) {
	initSketch(cms->ss, hash1);
	initSketch(cms->ss + 1, hash2);
	initSketch(cms->ss + 2, hash3);
	initSketch(cms->ss + 3, hash4);
}

// добавя елемента към всички Sketch-ове
void addCMS(CMSketch* cms, int v) {
	int i;
	for(i = 0; i < D; ++i) {
		addSkech(cms->ss + i, v);
	}
}

// връща минималната стойност на getCountSketch за всичките Sketch
Count getCountCMS(CMSketch* cms, int v) {
	Count c = getCountSketch(cms->ss, v);
	int i;
	for (i = 1; i < D; ++i) {
		Count next = getCountSketch(cms->ss + i, v);
		c = MIN(c, next);
	}
	return c;
}

/*
	-----------------------------------------------------------------------
	Реализация на скица за Heavy Hitters.
	-----------------------------------------------------------------------
*/

// Резултатът от алгоритъма
// Получаваме най-много к елемента, които се срещат поне M/k пъти
// elems е масив, съдържащ всички намерени елементи
// count - техният брой
typedef struct HeavyHitters {
	int* elems;
	unsigned int count;
} HeavyHitters;

void initHavyHitters(HeavyHitters* hh, unsigned int k) {
	hh->elems = (int*) malloc(k * sizeof(int));
	hh->count = 0;
}

void freeHeavyHitters(HeavyHitters* hh) {
	free(hh->elems);
	hh->count = 0;
}

// добавя нов елемент
void pushHeavyHitter(HeavyHitters* hh, int v) {
	hh->elems[hh->count++] = v;
}

/*
	Самият алгоритъм за често срещани елементи.
*/

typedef struct HHSketch {
	Count total;
	CMSketch cms[LOGN];
} HHSketch;

// инициализира празна скица
void initHHSketch(HHSketch* hhs) {
	int i;
	for (i = 0; i < LOGN; ++i) {
		initCMSketch(hhs->cms + i);
	}
}

// добавя нов елемент
void addHHSketch(HHSketch* hhs, int v) {
	// увеличаваме броя на всички елементш
	++hhs->total;

	// позиционираме се в лявото ил дясното поддърво
	// в зависимост от n
	int from, to;
	if (v > N / 2) {
		from = N/2;
		to = N;
	} else {
		from  = 0;
		to = N/2;
	}

	CMSketch* cur = hhs->cms;
	// само за тестване
	int height = 0;
	while(from + 1!=to) {
		assert(height < LOGN - 1);
		//printf("height:%d h\t\t from: %d\t\t to: %d\n", height, from, to);
		// Всеки интервал се асоциира с края си
		// Началото също е вариант, но краят е елемент от интервало и има симетрия с листата
		addCMS(cur, to);

		int mid = (to + from) / 2;
		if (v > mid) {
			// дясното поддърво
			from = mid;
		} else {
			// лявото поддърво
			to = mid;
		}
		++cur;
		++height;
	}
	// листото
	addCMS(cur, to);
}

// връща елементите, броят на срещанията на които е по-голям от @goal
void findHeavyHittersImpl(
	HeavyHitters* hh, Count goal,
	int from, int to, CMSketch* cms,
	int height /* за тестване */){
	assert(height < LOGN);
	// На тази височина се пази информация да интервалите (from; mid] (mid; to]
	int mid = (from + to) / 2;
	//printf("height:%d h\t\t from: %d\t\t mid: %d\t\t to: %d\n", height, from, mid, to);

	if(from + 1 == mid) {
		// листо
		if (getCountCMS(cms, mid) >= goal) pushHeavyHitter(hh, mid);
	} else if (getCountCMS(cms, mid) >= goal) {
		// лявото поддърво
		findHeavyHittersImpl(hh, goal, from, mid, cms + 1, height + 1);
	}


	if(mid + 1 == to) {
		// листо
		if (getCountCMS(cms, to) >= goal) pushHeavyHitter(hh, to);
	} else if (getCountCMS(cms, to) >= goal) {
		// дясното поддърво
		findHeavyHittersImpl(hh, goal, mid, to, cms + 1, height + 1);
	}
}

// връща всички емевнти, които се срещат повече он |M|/к пъти
HeavyHitters findHeavyHitters(HHSketch* hhs, unsigned int k) {
	HeavyHitters hh;
	initHavyHitters(&hh, k);

	Count goal = hhs->total / k;

	findHeavyHittersImpl(&hh, goal, 0, N, hhs->cms, 0);
	return hh;
}

/*
	--------------------------------------------------------------------------
	Тестове за функциолността.
	Това са нещо като юнит тестове, които ползвах при реализацията.
	За да не използвам библиотеки, съм ги написал по най-простия възможен начин.
	Реших да не ги трия, защото в някои от тях може да се види очакваното
	поведение. Можете да ги игнорирате.
	--------------------------------------------------------------------------
*/

void test_emptySketch() {
	Sketch sketch;
	initSketch(&sketch, hash1);

	assert(0 == getCountSketch(&sketch, 26));
}

void test_sameElementsSketch() {
	Sketch sketch;
	initSketch(&sketch, hash1);
	const int x = 1;
	const int n = 1000;
	int i;
	for(i = 0; i < n; ++ i) {
		addSkech(&sketch, x);
	}
	assert(n == getCountSketch(&sketch, x));
}

void test_emptyCMS() {
	CMSketch cms;
	initCMSketch(&cms);

	assert(0 == getCountCMS(&cms, 26));
}

void test_sameElementsCMS() {
	CMSketch cms;
	initCMSketch(&cms);

	const int x = 1;
	const int n = 1000;
	int i;
	for(i = 0; i < n; ++ i) {
		addCMS(&cms, x);
	}
	assert(n == getCountCMS(&cms, x));
}

void test_boundsCMS() {
	CMSketch cms;
	initCMSketch(&cms);

	const int n = 10000;
	const int x = 26;
	Count c = (Count)n;
	srand(time(0));
	int i;
	for (i = 0; i < n; ++i) {
		int a = random();
		if (a == x) ++c;
		addCMS(&cms, a);
		addCMS(&cms, x);
	}

	// оценката за f е по-голяма от f
	assert(c <= getCountCMS(&cms, x));
	//printf("%u\n", c);
	//printf("%u\n", getCountCMS(&cms, x));
}

void test_addHHS() {
	HHSketch hhs;
	initHHSketch(&hhs);
	addHHSketch(&hhs, 1);

	assert(getCountCMS(hhs.cms + LOGN - 1, 1) == 1);

	HeavyHitters hh = findHeavyHitters(&hhs, 1);
	assert(hh.count == 1);
	freeHeavyHitters(&hh);
}

void test_findHalfHeavyHitters() {
	HHSketch hhs;
	initHHSketch(&hhs);

	const int n = 10000;
	const int x = 1;
	srand(time(0));
	int i;
	for (i = 0; i < n; ++i) {
		int a = random() % N;
		addHHSketch(&hhs, a);
		addHHSketch(&hhs, x);
	}

	HeavyHitters hh = findHeavyHitters(&hhs, 2);

	// x e heavy hitter
	assert(hh.count >= 1);

	for(i = 0; i < hh.count; ++i) {
		if(hh.elems[i] == x) {
			freeHeavyHitters(&hh);
			return;
		}
	}
	assert(0);
}

void test_findHalfHeavyHittersLittleVariety() {
	// ще добавяме произволни числа от 1 до к
	const int k = 100;
	HHSketch hhs;
	initHHSketch(&hhs);

	const int n = 100000;
	srand(time(0));
	int i;
	for (i = 0; i < n; ++i) {
		int a = random() % k;
		addHHSketch(&hhs, a);
	}

	// От принципа на Дирихле има тъкъв елемент, който се среща 1/к пъти
	HeavyHitters hh = findHeavyHitters(&hhs, k);
	assert(hh.count);
	freeHeavyHitters(&hh);
}

// пуска всички тестове
void test() {
	test_emptySketch();
	test_sameElementsSketch();

	test_emptyCMS();
	test_sameElementsCMS();
	test_boundsCMS();
	test_addHHS();
	test_findHalfHeavyHitters();
	test_findHalfHeavyHittersLittleVariety();

	printf("Tests ran successfully!\n");
}


/*
 	Бенчмрк за алгоритъма.
*/

void initRandomArray(int* data, unsigned int size, unsigned int from, unsigned int to) {
		unsigned int i;
		for(i = 0; i < size; ++i) data[i] = rand() % to + from;
}

HeavyHitters getHeavyHitters(int* data, unsigned int size, unsigned int k) {
	HHSketch hhs;
	initHHSketch(&hhs);
	unsigned int i;
	for(i = 0; i < size; ++i) addHHSketch(&hhs, data[i]);

	return findHeavyHitters(&hhs, k);
}

void benchmark() {
	const unsigned int numberExpe = 100;
	const unsigned int n = 10;
	const unsigned int step = 300000;
	int* data = (int*) malloc(step * n * sizeof(int));
	unsigned int i, j;

	srand(time(0));

	for (i = 0; i < numberExpe; ++i) {
		for(j = 1; j <= n; ++j) {
			if (j > 1) printf("\t");
			unsigned int size = j * step;
			// Тъй като генерирането на голиме масиви от произволни числа е бавно
			// реших, да ползвам по-малки масиви, но числата да са в по-малък интервал.
			// Така често срещаните елементи няма да ас винаги 0 и сегментното дърво ще
			// се обхожда, което до някаква степен е и по-реалистично.
			// Много от елементите, които могат да попаднат в М, няма да са в него,
			// а ще има някакви елементи, които ще се срещета по няколко пъти.
			initRandomArray(data, size, 0 /* from */, 1000 /* to */);

			clock_t begin = clock();
			HeavyHitters hh = getHeavyHitters(data, size, 10 /* коефициентът к */);
			clock_t end = clock();
			unsigned long long timeElapsed = (unsigned long long)((double)(end - begin) / CLOCKS_PER_SEC * 100);

			freeHeavyHitters(&hh);
			printf("%llu", timeElapsed);
		}
		printf("\n");
	}
	free(data);
}

int main() {
	//test();

	benchmark();

	return 0;
}
