package net.asteasolutions.aos;

import java.util.ArrayList;
import java.util.Observable;
import java.util.Observer;

import android.content.Context;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Adapter;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;

/**
 * Custom {@link ArrayAdapter}, which adapts and displays our data in the custom views that the
 * {@link ListView} will be containing and showing. The adapter listens for eventual changes in the 
 * model by implementing the {@link Observer} interface, then being added as observer of the model.
 * When an {@link Adapter} itself notifies that its data has been changed, its {@link AdapterView} 
 * automatically updates its child views.
 */
class UberAdapter extends ArrayAdapter<ListItem> implements Observer {
    
    private final LayoutInflater inflater;
    
    /**
     * Constructor.
     * 
     * @param context a {@link Context}, used for creating the adapter and obtaining a reference to 
     * 					a {@link LayoutInflater} instance.
     * @param textViewResourceId the resource ID for a layout file, containing a {@link TextView} to
     * 		 			use when instantiating views. If we want to manually implement inflating and
     * 					initializing our own custom views, this parameter is irrelevant.
     * @param items the items that the UberAdapter will be "adapting".
     */
    public UberAdapter(Context context, int textViewResourceId, ArrayList<ListItem> items) {
        super(context, textViewResourceId, items);
        this.inflater = LayoutInflater.from(context);
    }
    
    /**
     * The most important method in {@link UberAdapter} (most of the cases - in any adapter).
     * It obtains an item at specific position from the adapter, then populates the specific view
     * that our {@link ListView} will be displaying. It can reuse already initialized views (ones 
     * which had previously displayed items) and doesn't create such new (heavy) objects, if it 
     * isn't necessary.
     */
    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
    	//View resultView = null;
        View resultView = convertView;
        
        // If nothing is passed as convertView, we still haven't initialized enough views to be able
        // to display an adapter view full of items plus one. In this case, inflate new view from 
        // our custom layout template.
        if (resultView == null) {
            resultView = this.inflater.inflate(R.layout.list_item, null);
            try {
            	// This is done just to emphasize the effect of initializing and garbage collecting
            	// a lot of heavy view objects.
                Thread.sleep(100);
            } catch (InterruptedException e) {
                Log.e(getClass().getSimpleName(), "", e);
            }
        }
        
        // Obtain the data at the specific position from the adapter.
        ListItem current = getItem(position);
        // Find the one view in our custom layout template, which will be displaying the data.
        // TODO Keep in mind that findViewById(int) is relatively heavy operation, so this can be 
        // further optimized by using the "ViewHolder" pattern.
        TextView titleView = (TextView) resultView.findViewById(R.id.title);
        
        if (titleView != null) {
            titleView.setText(current.getTitle());
        }
        
        return resultView;
    }
    
    @Override
    public int getViewTypeCount() {
    	// I tell the adapter how many buckets, caching "convertViews" this adapter should have.
    	return super.getViewTypeCount();
    }
    
    @Override
    public int getItemViewType(int position) {
    	// I tell the adapter in which bucket to look for "convertViews" for a given position.
    	return super.getItemViewType(position);
    }

    @Override
    public void update(Observable observable, Object data) {
    	// Makes the adapter view update its visible child views.
        notifyDataSetChanged();
    }
}
