package net.asteasolutions.aos;

import java.io.FileInputStream;
import java.io.IOException;

import android.app.Service;
import android.content.Intent;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.media.MediaPlayer.OnPreparedListener;
import android.os.Binder;
import android.os.IBinder;
import android.util.Log;

public class MusicPlayerService extends Service {
	private static final String LOG_TAG = "net.asteasolutions.aos";
	public static final String EXTRA_FILE_PATH = "filePath";

	private MediaPlayer mMediaPlayer;
	
	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		// We may receive multiple start commands after the service has been started. Here we
		// implement the following logic: If we don't have a MediaPlayer, create it and start
		// playback; otherwise resume playback.
		if (mMediaPlayer == null) {
			mMediaPlayer = new MediaPlayer();
			mMediaPlayer.setOnCompletionListener(new OnCompletionListener() {
				@Override
				public void onCompletion(MediaPlayer arg0) {
					stopSelf();
				}
			});
			mMediaPlayer.setOnPreparedListener(new OnPreparedListener() {
				@Override
				public void onPrepared(MediaPlayer mp) {
					mMediaPlayer.start();
				}
			});

			String filePath = intent.getStringExtra(EXTRA_FILE_PATH);
			try {
				FileInputStream fis = new FileInputStream(filePath);
				mMediaPlayer.setDataSource(fis.getFD());
				mMediaPlayer.prepareAsync();
			} catch (IllegalArgumentException e) {
				Log.e(LOG_TAG, "", e);
			} catch (IllegalStateException e) {
				Log.e(LOG_TAG, "", e);
			} catch (IOException e) {
				Log.e(LOG_TAG, "", e);
			}
		} else {
			mMediaPlayer.start();
		}
		
		return super.onStartCommand(intent, flags, startId);
	}
	
	@Override
	public void onDestroy() {
		super.onDestroy();
		if (mMediaPlayer != null) {
			mMediaPlayer.release();
		}
	}
	
	// The pause() and stop() methods represent the interface of the service. In this case these
	// methods are only available through a reference to the service, i.e. if someone binds to it. 
	public void pause() {
		if (mMediaPlayer != null && mMediaPlayer.isPlaying()) {
			mMediaPlayer.pause();
		}
	}
	
	public void stop() {
		if (mMediaPlayer != null) {
			mMediaPlayer.release();
			mMediaPlayer = null;
		}
		stopSelf();
	}
	
	
	@Override
	public IBinder onBind(Intent intent) {
		// When another component tries to bind to this service, we give it an instance of our
		// local Binder implementation. Implementing a custom Binder gives us control over what
		// exactly the "bond" between other components and this service will be.
		return new LocalBinder();
	}
	
	// This is our local Binder implementation. Through this Binder we decided to expose a reference
	// to the service itself. This is not always a good idea. Sometimes we may want to expose only
	// a part of the service's interface. In such cases we may use the Binder itself as a facade.
	public class LocalBinder extends Binder {
		public MusicPlayerService getService() {
			return MusicPlayerService.this;
		}
	}

}
