package net.asteasolutions.aos;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import net.asteasolutions.aos.MusicPlayerService.LocalBinder;
import android.app.Activity;
import android.content.ComponentName;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.IBinder;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.Toast;

public class WorkingInTheBackgroundActivity extends Activity {
	private static final String LOG_TAG = "net.asteasolutions.aos";
	private static final String FILE_NAME = "song.mp3";
	
	private EditText mAddressEditText;
	private Button mDownloadButton;
	private LinearLayout mPlayerControls;
	
	private MusicPlayerService mMusicService;
	private ServiceConnection mMusicServiceConnection;
	
    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        
        // Get references to the views we will use later
        mDownloadButton = (Button) findViewById(R.id.downloadButton);
        mAddressEditText = (EditText) findViewById(R.id.songAddress);
        mPlayerControls = (LinearLayout) findViewById(R.id.playerControls);
        if (!getFileStreamPath(FILE_NAME).exists()) {
        	mPlayerControls.setVisibility(View.GONE);
        }
        
    	// Create the service connection object. It will be used when we bind to the MusicService.
    	// When we call bindService() Android will establish the connection between the service
    	// and this activity by calling the connection object's onServiceConnected method
    	// (implemented below). We should use this method (more particularly its IBinder argument)
    	// to bind the activity to the service. "Binding" is an abstract term. Actually we define
    	// what the actual "bond" between the service and the activity is. In this case we obtain a 
    	// reference to the service.
        mMusicServiceConnection = new ServiceConnection() {

    		@Override
    		public void onServiceConnected(ComponentName componentName, IBinder binder) {
    			LocalBinder localBinder = (LocalBinder) binder;
    			mMusicService = localBinder.getService();
    		}
    		
    		@Override
    		public void onServiceDisconnected(ComponentName componentName) {
    			mMusicService = null;
    		}
    	};
    }
    
    @Override
    protected void onResume() {
		super.onResume();
		
		// We bind to the music service when the activity comes to the foreground
		// so we can control the playback via the service's methods
		Intent intent = new Intent("net.asteasolutions.aos.MUSIC_SERVICE");
		bindService(intent, mMusicServiceConnection, BIND_AUTO_CREATE);
    }
    
    @Override
    protected void onPause() {
    	super.onPause();
    	
    	// In most cases there should be symmetry - if we bind in onResume, we should unbind in
    	// onPause. The same rule holds for attaching listeners, registering BroadcastReceivers etc.
    	unbindService(mMusicServiceConnection);
    }
    
    public void download(View view) {
    	mMusicService.stop();
    	mDownloadButton.setEnabled(false);
    	mPlayerControls.setVisibility(View.GONE);
    	
    	String songUrl = mAddressEditText.getText().toString();
    	
    	// When starting an AsyncTask we just instantiate it and call execute(). All the other
    	// code is in the corresponding implementation (see DownloadTask below).
    	DownloadTask downloadTask = new DownloadTask();
    	downloadTask.execute(songUrl, FILE_NAME);
    }
    
    public void play(View view) {
    	// To start or resume the playback we will send a start command to the service.
    	// This way the service will be kept alive even if this activity dies and unbinds from it.
    	Intent intent = new Intent("net.asteasolutions.aos.MUSIC_SERVICE");
		intent.putExtra(MusicPlayerService.EXTRA_FILE_PATH, getFileStreamPath(FILE_NAME).getAbsolutePath());
		startService(intent);
    }
    
    public void pause(View view) {
    	mMusicService.pause();
    }

    public void stop(View view) {
    	mMusicService.stop();
    }
    
    // This is a pretty standard implementation of AsyncTask.
    // See AsyncTask's documentation for more info.
    private class DownloadTask extends AsyncTask<String, Integer, Boolean> {

		@Override
		protected Boolean doInBackground(String... args) {
			String address = args[0];
			String fileName = args[1];

			try {
				publishProgress(0);
				URL url = new URL(address);
				HttpURLConnection connection = (HttpURLConnection) url.openConnection();
				InputStream is = connection.getInputStream();
				
				FileOutputStream fos = openFileOutput(fileName, MODE_PRIVATE);
				
				int contentLength = connection.getContentLength();
				int readBytes = 0;
				
				byte[] buffer = new byte[1 << 13];
				int bufferLength;
				
				while ((bufferLength = is.read(buffer)) > 0) {
					fos.write(buffer, 0, bufferLength);
					readBytes += bufferLength;
					
					publishProgress((int) Math.round(readBytes * 100.0 / contentLength));
				}

				fos.flush();
				fos.close();
				return true;
				
			} catch (MalformedURLException e) {
				Log.e(LOG_TAG, "", e);
			} catch (IOException e) {
				Log.e(LOG_TAG, "", e);
			}
			return false;
		}
		
		@Override
		protected void onProgressUpdate(Integer... values) {
			int progress = values[0];
			mDownloadButton.setText(progress + "%");
		}
		
		@Override
		protected void onPostExecute(Boolean success) {
			mDownloadButton.setText(R.string.download);
			mDownloadButton.setEnabled(true);
			final int messageId;
			if (success) {
		    	mPlayerControls.setVisibility(View.VISIBLE);
		    	messageId = R.string.download_success;
			} else {
				messageId = R.string.download_failure;
			}
			Toast.makeText(WorkingInTheBackgroundActivity.this, messageId, Toast.LENGTH_LONG).show();
		}
    	
    }
}