/// Polynomial algorithm
/// for a special case of 3-SAT
/// by Trayan Gospodinov

#include <iostream>
#include <vector>
#include <queue>
#include <algorithm>
#include <fstream>
#include <stdexcept>

using namespace std;

struct Disjunct3;

short cnt;
vector<bool> disjuncts;
queue<Disjunct3> q;

struct Disjunct3
{
    short a, b, c;
    Disjunct3(short a=0, short b=0, short c=0): a(a<0?cnt-a:a), b(b<0?cnt-b:b), c(c<0?cnt-c:c)
    {
        if(this->b==this->c)this->c=0;
        if(this->a==this->b){this->b=this->c;this->c=0;}
        if(this->a<this->b || this->b<this->c) throw runtime_error("Disjunct3() called with a<b or b<c or a<c");
    }

    bool checkTrue(const vector<bool> &v)const
    {
        if(a<=cnt && v[a-1]) return true;
        if(a>cnt && !v[a-cnt-1]) return true;
        if(b<=cnt && v[b-1]) return true;
        if(b>cnt && !v[b-cnt-1]) return true;
        if(c<=cnt && v[c-1]) return true;
        if(c>cnt && !v[c-cnt-1]) return true;
        return false;
    }

    ///win speed for memory
    long long getCode()const
    {
        return a+2*cnt*(b+2*cnt*(long long)c);
    }

    short size()const
    {
        return (bool)a+(bool)b+(bool)c;
    }

    void print(bool nl=true)const
    {
        if(a && b && c)cout<<'('<<(a>cnt?(cnt-a):a)<<","<<(b>cnt?(cnt-b):b)<<","<<(c>cnt?(cnt-c):c)<<')'<<(nl?"\n":"");
        if(a && b && !c)cout<<'('<<(a>cnt?(cnt-a):a)<<","<<(b>cnt?(cnt-b):b)<<')'<<(nl?"\n":"");
        if(a && !b && !c)cout<<'('<<(a>cnt?(cnt-a):a)<<')'<<(nl?"\n":"");
        if(!a && !b && !c)cout<<"()"<<(nl?"\n":"");
    }
};

///cnt=3 a=1 -> !a=4
///cnt=3 a=5 -> !a=2
short neg(short a)
{
    if(a<=cnt)return a+cnt;
    return a-cnt;
}

///non-method version => constructor not called
long long getCode(short a=0, short b=0, short c=0)
{
    if(a<0)a=cnt-a;
    if(b<0)b=cnt-b;
    if(c<0)c=cnt-c;
    if(b==c)c=0;
    if(a==b){b=c;c=0;}

    if(a<b || b<c) throw runtime_error("getCode called with a<b or b<c or a<c");
    return a+2*cnt*(b+2*cnt*(long long)c);
}

///debug tool
Disjunct3 getDisjunct(long long code)
{
    if(!code) return Disjunct3();
    short a=code%(2*cnt);
    if(code && !a) a=2*cnt;
    code-=a;
    code/=(2*cnt);
    short b=code%(2*cnt);
    if(code && !b) b=2*cnt;
    code-=b;
    code/=(2*cnt);
    return Disjunct3(a, b, code);
}

bool hasContradiction(int a)
{
    if(a>cnt){a-=cnt;a*=-1;}
    return disjuncts[getCode(-a)];
}

void addDisjunct(const Disjunct3 &d)
{
    //cout<<"  Added: ";d.print();
    q.push(d);
    disjuncts[d.getCode()]=true;

    if(d.size()==1 && hasContradiction(d.a)) throw false;
}

bool fillFirstAvailable()
{
    for(short t=1; t<=cnt; t++)
     if(!disjuncts[getCode(t)] && !disjuncts[getCode(-t)])
     {
         addDisjunct(Disjunct3(t));
         return true;
     }
    return false;
}

///(a)(!a+b) -> (b)
void addToQueueSize12(short a)
{
    short na=neg(a);///!a

    ///1. !a<b
    for(short b=na+1; b<=2*cnt; b++)
     if(!disjuncts[getCode(b)] && disjuncts[getCode(b,na)])
      addDisjunct(Disjunct3(b));

    ///2. !a>b
    for(short b=1; b<na; b++)
     if(!disjuncts[getCode(b)] && disjuncts[getCode(na,b)])
      addDisjunct(Disjunct3(b));
}

///(a)(!a+b+c) -> (b+c)
void addToQueueSize13(short a)
{
    short na=neg(a);///!a

    ///1. !a smallest
    for(short b=na+2; b<=2*cnt; b++)
     for(short c=na+1; c<b; c++)
      if(!disjuncts[getCode(b,c)] && disjuncts[getCode(b,c,na)])
       addDisjunct(Disjunct3(b,c));

    ///2. !a in the middle
    for(short b=na+1; b<=2*cnt; b++)
     for(short c=1; c<na; c++)
      if(!disjuncts[getCode(b,c)] && disjuncts[getCode(b,na,c)])
       addDisjunct(Disjunct3(b,c));

    ///3. !a largest
    for(short b=2; b<na; b++)
     for(short c=1; c<b; c++)
      if(!disjuncts[getCode(b,c)] && disjuncts[getCode(na,b,c)])
       addDisjunct(Disjunct3(b,c));
}

///(a+b)(!a) -> (b)
void addToQueueSize21(short a, short b)
{
    ///1. (a+b)(!a) -> (b)
    if(!disjuncts[getCode(b)] && disjuncts[getCode(neg(a))]) addDisjunct(Disjunct3(b));
    ///2. (a+b)(!b) -> (a)
    if(!disjuncts[getCode(a)] && disjuncts[getCode(neg(b))]) addDisjunct(Disjunct3(a));
}

///(a+b)(!a+c) -> (b+c)
void addToQueueSize22(short a, short b)
{
    short na=neg(a);///!a
    short nb=neg(b);///!b

    ///1. (a+b)(!a+c) -> b+c
    ///1.1. c is largest (amongst !a and b)
    for(short c=max(na,b)+1; c<=2*cnt; c++)
     if(!disjuncts[getCode(c,b)] && disjuncts[getCode(c,na)])
      addDisjunct(Disjunct3(c,b));
    ///1.2. c is smallest (amongst !a and b)
    short sm=min(na,b);
    for(short c=1; c<sm; c++)
     if(!disjuncts[getCode(b,c)] && disjuncts[getCode(na,c)])
      addDisjunct(Disjunct3(b,c));
    ///1.3. !a<c<=b
    if(na<b)
     for(short c=na+1; c<=b; c++)
      if(!disjuncts[getCode(b,c)] && disjuncts[getCode(c,na)])
       addDisjunct(Disjunct3(b,c));
    ///1.4. b<=c<!a
    if(b<na)
     for(short c=b; c<na; c++)
      if(!disjuncts[getCode(c,b)] && disjuncts[getCode(na,c)])
       addDisjunct(Disjunct3(c,b));

    ///2. (a+b)(!b+c) -> a+c
    ///2.1. c is largest (amongst !b and a)
    for(short c=max(nb,a)+1; c<=2*cnt; c++)
     if(!disjuncts[getCode(c,a)] && disjuncts[getCode(c,nb)])
      addDisjunct(Disjunct3(c,a));
    ///2.2. c is smallest (amongst !b and a)
    sm=min(nb,a);
    for(short c=1; c<sm; c++)
     if(!disjuncts[getCode(a,c)] && disjuncts[getCode(nb,c)])
      addDisjunct(Disjunct3(a,c));
    ///2.3. !b<c<=a
    if(nb<a)
     for(short c=nb+1; c<=a; c++)
      if(!disjuncts[getCode(a,c)] && disjuncts[getCode(c,nb)])
       addDisjunct(Disjunct3(a,c));
    ///2.4. a<=c<!b
    if(a<nb)
     for(short c=a; c<nb; c++)
      if(!disjuncts[getCode(c,a)] && disjuncts[getCode(nb,c)])
       addDisjunct(Disjunct3(c,a));
}

///(a+b)(!a+c+d) -> (b+c+d)
void addToQueueSize23(short a, short b)
{
    short na=neg(a);
    short nb=neg(b);

    short sm=min(na,b);
    short bi=max(na,b);

    ///1. (a+b)(!a+c+d) -> (b+c+d)
    ///1.1. c and d are largest (amongst !a and b)
    for(short c=bi+2; c<=2*cnt; c++)
     for(short d=bi+1; d<c; d++)
      if(!disjuncts[getCode(c,d,b)] && disjuncts[getCode(c,d,na)])
       addDisjunct(Disjunct3(c,d,b));
    ///1.2. c and d are smallest (amongst !a and b)
    for(short c=2; c<sm; c++)
     for(short d=1; d<c; d++)
      if(!disjuncts[getCode(b,c,d)] && disjuncts[getCode(na,c,d)])
       addDisjunct(Disjunct3(b,c,d));
    ///1.3. d<{!a,b}<c
    for(short c=bi+1; c<=2*cnt; c++)
     for(short d=1; d<sm; d++)
      if(!disjuncts[getCode(c,b,d)] && disjuncts[getCode(c,na,d)])
       addDisjunct(Disjunct3(c,b,d));
    ///1.4. !a<b
    if(na<b)
    {
        ///1.4.1. d<!a<c<=b
        for(short c=na+1; c<=b; c++)
         for(short d=1; d<na; d++)
          if(!disjuncts[getCode(b,c,d)] && disjuncts[getCode(c,na,d)])
           addDisjunct(Disjunct3(b,c,d));
        ///1.4.2. !a<d<c<=b
        for(short c=na+2; c<=b; c++)
         for(short d=na+1; d<c; d++)
          if(!disjuncts[getCode(b,c,d)] && disjuncts[getCode(c,d,na)])
           addDisjunct(Disjunct3(b,c,d));
        ///1.4.3. !a<d<=b<=c
        for(short c=b; c<=2*cnt; c++)
         for(short d=na+1; d<=b; d++)
          if(!disjuncts[getCode(c,b,d)] && disjuncts[getCode(c,d,na)])
           addDisjunct(Disjunct3(c,b,d));
    }
    ///1.5. b<!a
    if(b<na)
    {
        ///1.5.1. d<=b<=c<!a
        for(short c=b; c<na; c++)
         for(short d=1; d<=b; d++)
          if(!disjuncts[getCode(c,b,d)] && disjuncts[getCode(na,c,d)])
           addDisjunct(Disjunct3(c,b,d));
        ///1.5.2. b<=d<c<!a
        for(short c=b+1; c<na; c++)
         for(short d=b; d<c; d++)
          if(!disjuncts[getCode(c,d,b)] && disjuncts[getCode(na,c,d)])
           addDisjunct(Disjunct3(c,d,b));
        ///1.5.3. b<=d<!a<c
        for(short c=na+1; c<=2*cnt; c++)
         for(short d=b; d<na; d++)
          if(!disjuncts[getCode(c,d,b)] && disjuncts[getCode(c,na,d)])
           addDisjunct(Disjunct3(c,d,b));
    }

    sm=min(nb,a);
    bi=max(nb,a);

    ///2. (a+b)(!b+c+d) -> (a+c+d)
    ///2.1. c and d are largest (amongst !b and a)
    for(short c=bi+2; c<=2*cnt; c++)
     for(short d=bi+1; d<c; d++)
      if(!disjuncts[getCode(c,d,a)] && disjuncts[getCode(c,d,nb)])
       addDisjunct(Disjunct3(c,d,a));
    ///2.2. c and d are smallest (amongst !b and a)
    for(short c=2; c<sm; c++)
     for(short d=1; d<c; d++)
      if(!disjuncts[getCode(a,c,d)] && disjuncts[getCode(nb,c,d)])
       addDisjunct(Disjunct3(a,c,d));
    ///2.3. d<{!b,a}<c
    for(short c=bi+1; c<=2*cnt; c++)
     for(short d=1; d<sm; d++)
      if(!disjuncts[getCode(c,a,d)] && disjuncts[getCode(c,nb,d)])
       addDisjunct(Disjunct3(c,a,d));
    ///2.4. !b<a
    if(nb<a)
    {
        ///2.4.1. d<!b<c<=a
        for(short c=nb+1; c<=a; c++)
         for(short d=1; d<nb; d++)
          if(!disjuncts[getCode(a,c,d)] && disjuncts[getCode(c,nb,d)])
           addDisjunct(Disjunct3(a,c,d));
        ///2.4.2. !b<d<c<=a
        for(short c=nb+2; c<=a; c++)
         for(short d=nb+1; d<c; d++)
          if(!disjuncts[getCode(a,c,d)] && disjuncts[getCode(c,d,nb)])
           addDisjunct(Disjunct3(a,c,d));
        ///2.4.3. !b<d<=a<=c
        for(short c=a; c<=2*cnt; c++)
         for(short d=nb+1; d<=a; d++)
          if(!disjuncts[getCode(c,a,d)] && disjuncts[getCode(c,d,nb)])
           addDisjunct(Disjunct3(c,a,d));
    }
    ///2.5. a<!b
    if(b<na)
    {
        ///2.5.1. d<=a<=c<!b
        for(short c=a; c<nb; c++)
         for(short d=1; d<=a; d++)
          if(!disjuncts[getCode(c,a,d)] && disjuncts[getCode(nb,c,d)])
           addDisjunct(Disjunct3(c,a,d));
        ///2.5.2. a<=d<c<!b
        for(short c=a+1; c<nb; c++)
         for(short d=a; d<c; d++)
          if(!disjuncts[getCode(c,d,a)] && disjuncts[getCode(nb,c,d)])
           addDisjunct(Disjunct3(c,d,a));
        ///2.5.3. a<=d<!b<c
        for(short c=nb+1; c<=2*cnt; c++)
         for(short d=a; d<nb; d++)
          if(!disjuncts[getCode(c,d,a)] && disjuncts[getCode(c,nb,d)])
           addDisjunct(Disjunct3(c,d,a));
    }
}

///(a+b+c)(!a) -> (b+c)
void addToQueueSize31(short a, short b, short c)
{
    ///1. (a+b+c)(!a) -> (b+c)
    if(!disjuncts[getCode(b,c)] && disjuncts[getCode(neg(a))]) addDisjunct(Disjunct3(b,c));
    ///2. (a+b+c)(!b) -> (a+c)
    if(!disjuncts[getCode(a,c)] && disjuncts[getCode(neg(b))]) addDisjunct(Disjunct3(a,c));
    ///2. (a+b+c)(!c) -> (a+b)
    if(!disjuncts[getCode(a,b)] && disjuncts[getCode(neg(c))]) addDisjunct(Disjunct3(a,b));

}

///(a+b+c)(!a+d) -> (b+c+d)
void addToQueueSize32(short a, short b, short c)
{
    short na=neg(a);
    short nb=neg(b);
    short nc=neg(c);

    short sm=min(na,c);///b>c
    short bi=max(na,b);///b>c

    ///1. (a+b+c)(!a+d) -> (b+c+d)
    ///1.1. d is smallest (amongst !a, b and c)
    for(short d=1; d<sm; d++)
     if(!disjuncts[getCode(b,c,d)] && disjuncts[getCode(na,d)])
      addDisjunct(Disjunct3(b,c,d));
    ///1.2. d is largest (amongst !a, b and c)
    for(short d=bi+1; d<=2*cnt; d++)
     if(!disjuncts[getCode(d,b,c)] && disjuncts[getCode(d,na)])
      addDisjunct(Disjunct3(d,b,c));
    ///1.3. !a>b>c
    if(na>b)
    {
        ///1.3.1. !a>d>=b>c
        for(short d=b; d<na; d++)
         if(!disjuncts[getCode(d,b,c)] && disjuncts[getCode(na,d)])
          addDisjunct(Disjunct3(d,b,c));
        ///1.3.2. !a>b>d>=c
        for(short d=c; d<b; d++)
         if(!disjuncts[getCode(b,d,c)] && disjuncts[getCode(na,d)])
          addDisjunct(Disjunct3(b,d,c));
    }
    ///1.4. b>!a>c
    else if(na>c)
    {
        ///1.4.1. b>=d>!a>c
        for(short d=na+1; d<=b; d++)
         if(!disjuncts[getCode(b,d,c)] && disjuncts[getCode(d,na)])
          addDisjunct(Disjunct3(b,d,c));
        ///1.4.2. b>!a>d>=c
        for(short d=c; d<na; d++)
         if(!disjuncts[getCode(b,d,c)] && disjuncts[getCode(na,d)])
          addDisjunct(Disjunct3(b,d,c));
    }
    ///1.5. b>c>!a
    else
    {
        ///1.5.1. b>=d>c>!a
        for(short d=c+1; d<=b; d++)
         if(!disjuncts[getCode(b,d,c)] && disjuncts[getCode(d,na)])
          addDisjunct(Disjunct3(b,d,c));
        ///1.5.2. b>c>=d>!a
        for(short d=na+1; d<=c; d++)
         if(!disjuncts[getCode(b,c,d)] && disjuncts[getCode(d,na)])
          addDisjunct(Disjunct3(b,c,d));
    }

    sm=min(nb,c);///a>c
    bi=max(nb,a);///a>c

    ///2. (a+b+c)(!b+d) -> (a+c+d)
    ///2.1. d is smallest (amongst a, !b and c)
    for(short d=1; d<sm; d++)
     if(!disjuncts[getCode(a,c,d)] && disjuncts[getCode(nb,d)])
      addDisjunct(Disjunct3(a,c,d));
    ///2.2. d is largest (amongst a, !b and c)
    for(short d=bi+1; d<=2*cnt; d++)
     if(!disjuncts[getCode(d,a,c)] && disjuncts[getCode(d,nb)])
      addDisjunct(Disjunct3(d,a,c));
    ///2.3. !b>a>c
    if(nb>a)
    {
        ///2.3.1. !b>d>=a>c
        for(short d=a; d<nb; d++)
         if(!disjuncts[getCode(d,a,c)] && disjuncts[getCode(nb,d)])
          addDisjunct(Disjunct3(d,a,c));
        ///2.3.2. !b>a>d>=c
        for(short d=c; d<a; d++)
         if(!disjuncts[getCode(a,d,c)] && disjuncts[getCode(nb,d)])
          addDisjunct(Disjunct3(a,d,c));
    }
    ///2.4. a>!b>c
    else if(nb>c)
    {
        ///2.4.1. a>=d>!b>c
        for(short d=nb+1; d<=a; d++)
         if(!disjuncts[getCode(a,d,c)] && disjuncts[getCode(d,nb)])
          addDisjunct(Disjunct3(a,d,c));
        ///2.4.2. a>!b>d>=c
        for(short d=c; d<nb; d++)
         if(!disjuncts[getCode(a,d,c)] && disjuncts[getCode(nb,d)])
          addDisjunct(Disjunct3(a,d,c));
    }
    ///2.5. a>c>!b
    else
    {
        ///2.5.1. a>=d>c>!b
        for(short d=c+1; d<=a; d++)
         if(!disjuncts[getCode(a,d,c)] && disjuncts[getCode(d,nb)])
          addDisjunct(Disjunct3(a,d,c));
        ///2.5.2. a>c>=d>!b
        for(short d=nb+1; d<=c; d++)
         if(!disjuncts[getCode(a,c,d)] && disjuncts[getCode(d,nb)])
          addDisjunct(Disjunct3(a,c,d));
    }

    sm=min(nc,b);///a>b
    bi=max(nc,a);///a>b

    ///3. (a+b+c)(!c+d) -> (a+b+d)
    ///3.1. d is smallest (amongst a, b and !c)
    for(short d=1; d<sm; d++)
     if(!disjuncts[getCode(a,b,d)] && disjuncts[getCode(nc,d)])
      addDisjunct(Disjunct3(a,b,d));
    ///3.2. d is largest (amongst a, b and !c)
    for(short d=bi+1; d<=2*cnt; d++)
     if(!disjuncts[getCode(d,a,b)] && disjuncts[getCode(d,nc)])
      addDisjunct(Disjunct3(d,a,b));
    ///3.3. !c>a>b
    if(nc>a)
    {
        ///3.3.1. !c>d>=a>b
        for(short d=a; d<nc; d++)
         if(!disjuncts[getCode(d,a,b)] && disjuncts[getCode(nc,d)])
          addDisjunct(Disjunct3(d,a,b));
        ///3.3.2. !c>a>d>=b
        for(short d=b; d<a; d++)
         if(!disjuncts[getCode(a,d,b)] && disjuncts[getCode(nc,d)])
          addDisjunct(Disjunct3(a,d,b));
    }
    ///3.4. a>!c>b
    else if(nc>b)
    {
        ///3.4.1. a>=d>!c>b
        for(short d=nc+1; d<=a; d++)
         if(!disjuncts[getCode(a,d,b)] && disjuncts[getCode(d,nc)])
          addDisjunct(Disjunct3(a,d,b));
        ///3.4.2. a>!c>d>=b
        for(short d=b; d<nc; d++)
         if(!disjuncts[getCode(a,d,b)] && disjuncts[getCode(nc,d)])
          addDisjunct(Disjunct3(a,d,b));
    }
    ///3.5. a>b>!c
    else
    {
        ///3.5.1. a>=d>b>!c
        for(short d=b+1; d<=a; d++)
         if(!disjuncts[getCode(a,d,b)] && disjuncts[getCode(d,nc)])
          addDisjunct(Disjunct3(a,d,b));
        ///3.5.2. a>b>=d>!c
        for(short d=nc+1; d<=b; d++)
         if(!disjuncts[getCode(a,b,d)] && disjuncts[getCode(d,nc)])
          addDisjunct(Disjunct3(a,b,d));
    }
}

///(a+b+c)(!a+b+d) -> (b+c+d)
void addToQueueSize33(short a, short b, short c)
{
    short na=neg(a);
    short nb=neg(b);
    short nc=neg(c);

    ///1. (a+b+c)[(!a+b+d) + (!a+c+d)] -> (b+c+d)
    ///1.1. !a>b>c
    if(na>b)
    {
        ///1.1.1. d>!a>b>c
        for(short d=na+1; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,b,c)] && (disjuncts[getCode(d,na,b)] || disjuncts[getCode(d,na,c)]))
          addDisjunct(Disjunct3(d,b,c));
        ///1.1.2. !a>d>=b>c
        for(short d=b; d<na; d++)
         if(!disjuncts[getCode(d,b,c)] && (disjuncts[getCode(na,d,b)] || disjuncts[getCode(na,d,c)]))
          addDisjunct(Disjunct3(d,b,c));
        ///1.1.3. !a>b>d>=c
        for(short d=c; d<b; d++)
         if(!disjuncts[getCode(b,d,c)] && (disjuncts[getCode(na,b,d)] || disjuncts[getCode(na,d,c)]))
          addDisjunct(Disjunct3(b,d,c));
        ///1.1.4. !a>b>c>d
        for(short d=1; d<c; d++)
         if(!disjuncts[getCode(b,c,d)] && (disjuncts[getCode(na,b,d)] || disjuncts[getCode(na,c,d)]))
          addDisjunct(Disjunct3(b,c,d));
    }
    ///1.2. b>!a>c
    else if(na>c)
    {
        ///1.2.1. d>=b>!a>c
        for(short d=b; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,b,c)] && (disjuncts[getCode(d,b,na)] || disjuncts[getCode(d,na,c)]))
          addDisjunct(Disjunct3(d,b,c));
        ///1.2.2. b>d>!a>c
        for(short d=na+1; d<b; d++)
         if(!disjuncts[getCode(b,d,c)] && (disjuncts[getCode(b,d,na)] || disjuncts[getCode(d,na,c)]))
          addDisjunct(Disjunct3(b,d,c));
        ///1.2.3. b>!a>d>=c
        for(short d=c; d<na; d++)
         if(!disjuncts[getCode(b,d,c)] && (disjuncts[getCode(b,na,d)] || disjuncts[getCode(na,d,c)]))
          addDisjunct(Disjunct3(b,d,c));
        ///1.2.4. b>!a>c>d
        for(short d=1; d<c; d++)
         if(!disjuncts[getCode(b,c,d)] && (disjuncts[getCode(b,na,d)] || disjuncts[getCode(na,c,d)]))
          addDisjunct(Disjunct3(b,c,d));
    }
    ///1.3. b>c>!a
    else
    {
        ///1.3.1. d>=b>c>!a
        for(short d=b; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,b,c)] && (disjuncts[getCode(d,b,na)] || disjuncts[getCode(d,c,na)]))
          addDisjunct(Disjunct3(d,b,c));
        ///1.3.2. b>d>=c>!a
        for(short d=c; d<b; d++)
         if(!disjuncts[getCode(b,d,c)] && (disjuncts[getCode(b,d,na)] || disjuncts[getCode(d,c,na)]))
          addDisjunct(Disjunct3(b,d,c));
        ///1.3.3. b>c>d>!a
        for(short d=na+1; d<c; d++)
         if(!disjuncts[getCode(b,c,d)] && (disjuncts[getCode(b,d,na)] || disjuncts[getCode(c,d,na)]))
          addDisjunct(Disjunct3(b,c,d));
        ///1.3.4. b>c>!a>d
        for(short d=1; d<na; d++)
         if(!disjuncts[getCode(b,c,d)] && (disjuncts[getCode(b,na,d)] || disjuncts[getCode(c,na,d)]))
          addDisjunct(Disjunct3(b,c,d));
    }

    ///2. (a+b+c)[(!b+a+d) + (!b+c+d)] -> (a+c+d)
    ///2.1. !b>a>c
    if(nb>a)
    {
        ///2.1.1. d>!b>a>c
        for(short d=nb+1; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,c)] && (disjuncts[getCode(d,nb,a)] || disjuncts[getCode(d,nb,c)]))
          addDisjunct(Disjunct3(d,a,c));
        ///2.1.2. !b>d>=a>c
        for(short d=a; d<nb; d++)
         if(!disjuncts[getCode(d,a,c)] && (disjuncts[getCode(nb,d,a)] || disjuncts[getCode(nb,d,c)]))
          addDisjunct(Disjunct3(d,a,c));
        ///2.1.3. !b>a>d>=c
        for(short d=c; d<a; d++)
         if(!disjuncts[getCode(a,d,c)] && (disjuncts[getCode(nb,a,d)] || disjuncts[getCode(nb,d,c)]))
          addDisjunct(Disjunct3(a,d,c));
        ///2.1.4. !b>a>c>d
        for(short d=1; d<c; d++)
         if(!disjuncts[getCode(a,c,d)] && (disjuncts[getCode(nb,a,d)] || disjuncts[getCode(nb,c,d)]))
          addDisjunct(Disjunct3(a,c,d));
    }
    ///2.2. a>!b>c
    else if(nb>c)
    {
        ///2.2.1. d>=a>!b>c
        for(short d=a; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,c)] && (disjuncts[getCode(d,a,nb)] || disjuncts[getCode(d,nb,c)]))
          addDisjunct(Disjunct3(d,a,c));
        ///2.2.2. a>d>!b>c
        for(short d=nb+1; d<a; d++)
         if(!disjuncts[getCode(a,d,c)] && (disjuncts[getCode(a,d,nb)] || disjuncts[getCode(d,nb,c)]))
          addDisjunct(Disjunct3(a,d,c));
        ///2.2.3. a>!b>d>=c
        for(short d=c; d<nb; d++)
         if(!disjuncts[getCode(a,d,c)] && (disjuncts[getCode(a,nb,d)] || disjuncts[getCode(nb,d,c)]))
          addDisjunct(Disjunct3(a,d,c));
        ///2.2.4. a>!b>c>d
        for(short d=1; d<c; d++)
         if(!disjuncts[getCode(a,c,d)] && (disjuncts[getCode(a,nb,d)] || disjuncts[getCode(nb,c,d)]))
          addDisjunct(Disjunct3(a,c,d));
    }
    ///2.3. a>c>!b
    else
    {
        ///2.3.1. d>=a>c>!b
        for(short d=a; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,c)] && (disjuncts[getCode(d,a,nb)] || disjuncts[getCode(d,c,nb)]))
          addDisjunct(Disjunct3(d,a,c));
        ///2.3.2. a>d>=c>!b
        for(short d=c; d<a; d++)
         if(!disjuncts[getCode(a,d,c)] && (disjuncts[getCode(a,d,nb)] || disjuncts[getCode(d,c,nb)]))
          addDisjunct(Disjunct3(a,d,c));
        ///2.3.3. a>c>d>!b
        for(short d=nb+1; d<c; d++)
         if(!disjuncts[getCode(a,c,d)] && (disjuncts[getCode(a,d,nb)] || disjuncts[getCode(c,d,nb)]))
          addDisjunct(Disjunct3(a,c,d));
        ///2.3.4. a>c>!b>d
        for(short d=1; d<nb; d++)
         if(!disjuncts[getCode(a,c,d)] && (disjuncts[getCode(a,nb,d)] || disjuncts[getCode(c,nb,d)]))
          addDisjunct(Disjunct3(a,c,d));
    }

    ///3. (a+b+c)[(!c+a+d) + (!c+b+d)] -> (a+b+d)
    ///3.1. !c>a>b
    if(nc>a)
    {
        ///3.1.1. d>!c>a>b
        for(short d=nc+1; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,b)] && (disjuncts[getCode(d,nc,a)] || disjuncts[getCode(d,nc,b)]))
          addDisjunct(Disjunct3(d,a,b));
        ///3.1.2. !c>d>=a>b
        for(short d=a; d<nc; d++)
         if(!disjuncts[getCode(d,a,b)] && (disjuncts[getCode(nc,d,a)] || disjuncts[getCode(nc,d,b)]))
          addDisjunct(Disjunct3(d,a,b));
        ///3.1.3. !c>a>d>=b
        for(short d=b; d<a; d++)
         if(!disjuncts[getCode(a,d,b)] && (disjuncts[getCode(nc,a,d)] || disjuncts[getCode(nc,d,b)]))
          addDisjunct(Disjunct3(a,d,b));
        ///3.1.4. !c>a>b>d
        for(short d=1; d<b; d++)
         if(!disjuncts[getCode(a,b,d)] && (disjuncts[getCode(nc,a,d)] || disjuncts[getCode(nc,b,d)]))
          addDisjunct(Disjunct3(a,b,d));
    }
    ///3.2. a>!c>b
    else if(nc>b)
    {
        ///3.2.1. d>=a>!c>b
        for(short d=a; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,b)] && (disjuncts[getCode(d,a,nc)] || disjuncts[getCode(d,nc,b)]))
          addDisjunct(Disjunct3(d,a,b));
        ///3.2.2. a>d>!c>b
        for(short d=nc+1; d<a; d++)
         if(!disjuncts[getCode(a,d,b)] && (disjuncts[getCode(a,d,nc)] || disjuncts[getCode(d,nc,b)]))
          addDisjunct(Disjunct3(a,d,b));
        ///3.2.3. a>!c>d>=b
        for(short d=b; d<nc; d++)
         if(!disjuncts[getCode(a,d,b)] && (disjuncts[getCode(a,nc,d)] || disjuncts[getCode(nc,d,b)]))
          addDisjunct(Disjunct3(a,d,b));
        ///3.2.4. a>!c>b>d
        for(short d=1; d<b; d++)
         if(!disjuncts[getCode(a,b,d)] && (disjuncts[getCode(a,nc,d)] || disjuncts[getCode(nc,b,d)]))
          addDisjunct(Disjunct3(a,b,d));
    }
    ///3.3. a>b>!c
    else
    {
        ///3.3.1. d>=a>b>!c
        for(short d=a; d<=2*cnt; d++)
         if(!disjuncts[getCode(d,a,b)] && (disjuncts[getCode(d,a,nc)] || disjuncts[getCode(d,b,nc)]))
          addDisjunct(Disjunct3(d,a,b));
        ///3.3.2. a>d>=b>!c
        for(short d=b; d<a; d++)
         if(!disjuncts[getCode(a,d,b)] && (disjuncts[getCode(a,d,nc)] || disjuncts[getCode(d,b,nc)]))
          addDisjunct(Disjunct3(a,d,b));
        ///3.3.3. a>b>d>!c
        for(short d=nc+1; d<b; d++)
         if(!disjuncts[getCode(a,b,d)] && (disjuncts[getCode(a,d,nc)] || disjuncts[getCode(b,d,nc)]))
          addDisjunct(Disjunct3(a,b,d));
        ///3.3.4. a>b>!c>d
        for(short d=1; d<nc; d++)
         if(!disjuncts[getCode(a,b,d)] && (disjuncts[getCode(a,nc,d)] || disjuncts[getCode(b,nc,d)]))
          addDisjunct(Disjunct3(a,b,d));
    }
}

void addToQueueSize1(short a)
{
    addToQueueSize12(a);
    addToQueueSize13(a);
}

void addToQueueSize2(short a, short b)
{
    addToQueueSize21(a,b);
    addToQueueSize22(a,b);
    addToQueueSize23(a,b);
}

void addToQueueSize3(short a, short b, short c)
{
    addToQueueSize32(a,b,c);
    addToQueueSize31(a,b,c);
    addToQueueSize33(a,b,c);
}

void addToQueue(const Disjunct3 &d)
{
    if(!d.size())throw runtime_error("Empty disjunct should have been caught earlier!");
    else if(d.size()==1)addToQueueSize1(d.a);
    else if(d.size()==2)addToQueueSize2(d.a,d.b);
    else addToQueueSize3(d.a,d.b,d.c);
}

void printDisjuncts(bool full=false)
{
    if(full)
    {
        for(int i=1;i<disjuncts.size();i++)
         if(disjuncts[i])
          getDisjunct(i).print();
    }
    else
    {
        for(int i=1;i<=2*cnt;i++)
         if(disjuncts[i]) getDisjunct(i).print();
    }
}

vector<Disjunct3> humanInput()
{
    cin>>cnt;
    vector<Disjunct3> disjuncts;
    short a,b,c;
    while(true)
    {
        cin>>a>>b>>c;
        if(a<0)a=cnt-a;
        if(b<0)b=cnt-b;
        if(c<0)c=cnt-c;
        if(a<b)swap(a,b);
        if(a<c)swap(a,c);
        if(b<c)swap(b,c);
        if(!a && !b && !c) break;
        disjuncts.push_back(Disjunct3(a,b,c));
    }
    return disjuncts;
}

/// -1 = unknown
///  0 = CNF is unsatisfiable
/// +1 = CNF is satisfiable
int solve(const vector<Disjunct3> &input)
{
    bool hasArtificialAtoms = false;
    try
    {
        for(int i=0; i<input.size(); i++) addDisjunct(input[i]);
        //cout<<endl;
        do
        {
            while(!q.empty())
            {
                //cout<<"Making pairs with ";q.front().print();
                addToQueue(q.front());
                //cout<<"Popped: ";q.front().print();cout<<endl;
                q.pop();
            }
            hasArtificialAtoms = true;
        }while(fillFirstAvailable());
    }
    catch(bool f)
    {
        //while(!q.empty()) q.pop();
        if(!hasArtificialAtoms) return 0;
        return -1;
    }

    vector<bool> ans(cnt);
    for(int i=1; i<=cnt; i++) if(disjuncts[getCode(i)]) ans[i-1]=true;
    for(int i=0; i<input.size(); i++) if(!input[i].checkTrue(ans)) return -1;

    return 1;
}

int main()
{
    vector<Disjunct3> input=humanInput();
    disjuncts.resize(8*cnt*cnt*cnt+4*cnt*cnt+2*cnt+1);

    int res=solve(input);
    if(res==-1)cout<<"Unknown\n";
    if(res==0)cout<<"No solution\n";
    if(res==1) printDisjuncts();

    return 0;
}
