import { Router, Response } from "express";
import HttpStatusCodes from "http-status-codes";

import { auth } from "../../middleware/auth";
import Profile from "../../models/Profile";
import User from "../../models/User";
import { TypedRequestBody } from "../../types";
import { RequestWithUserId } from "../../types/Request";

const router = Router();

interface PostProfileBody {
  firstName: string;
  lastName: string;
  username: string;
}

// @route   GET api/profile/me
// @desc    Get current user's profile
// @access  Private
router.get("/me", auth, async (req: RequestWithUserId, res: Response) => {
  try {
    const profile = await Profile.findOne({
      user: req.userId,
    }).populate("user", ["email"]);
    if (!profile) {
      return res.status(HttpStatusCodes.BAD_REQUEST).json({
        errors: [
          {
            msg: "There is no profile for this user",
          },
        ],
      });
    }

    res.json(profile);
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

// @route   POST api/profile
// @desc    Create or update user's profile
// @access  Private
router.post(
  "/",
  auth,
  async (req: TypedRequestBody<PostProfileBody>, res: Response) => {
    // validate
    const { firstName, lastName, username } = req.body;

    // Build profile object based on IProfile
    const profileFields = {
      user: req.userId,
      firstName,
      lastName,
      username,
    };

    try {
      let user = await User.findOne({ _id: req.userId });

      if (!user) {
        return res.status(HttpStatusCodes.BAD_REQUEST).json({
          errors: [
            {
              msg: "User not registered",
            },
          ],
        });
      }

      let profile = await Profile.findOne({ user: req.userId });
      if (profile) {
        // Update
        profile = await Profile.findOneAndUpdate(
          { user: req.userId },
          { $set: profileFields },
          { new: true }
        );

        return res.json(profile);
      }

      // Create
      profile = new Profile(profileFields);

      await profile.save();

      res.json(profile);
    } catch (err) {
      console.error((err as any).message);
      res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
    }
  }
);

// @route   GET api/profile
// @desc    Get all profiles
// @access  Public
router.get("/", async (_: RequestWithUserId, res: Response) => {
  try {
    const profiles = await Profile.find().populate("user", ["email"]);
    res.json(profiles);
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

// @route   GET api/profile/user/:userId
// @desc    Get profile by userId
// @access  Public
router.get("/user/:userId", async (req: RequestWithUserId, res: Response) => {
  try {
    const profile = await Profile.findOne({
      user: req.params.userId,
    }).populate("user", ["email"]);

    if (!profile) {
      return res
        .status(HttpStatusCodes.NOT_FOUND)
        .json({ msg: "Profile not found" });
    }
    res.json(profile);
  } catch (err) {
    console.error((err as any).message);
    if ((err as any).kind === "ObjectId") {
      return res
        .status(HttpStatusCodes.NOT_FOUND)
        .json({ msg: "Profile not found" });
    }
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

// @route   DELETE api/profile
// @desc    Delete profile and user
// @access  Private
router.delete("/", auth, async (req: RequestWithUserId, res: Response) => {
  try {
    // Remove profile
    await Profile.findOneAndRemove({ user: req.userId });
    // Remove user
    await User.findOneAndRemove({ _id: req.userId });

    res.json({ msg: "User removed" });
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

export default router;
