import bcrypt from "bcryptjs";
import { Router, Response } from "express";
import HttpStatusCodes from "http-status-codes";
import jwt from "jsonwebtoken";

import { auth } from "../../middleware/auth";
import Payload from "../../types/Payload";
import { RequestWithUserId } from "../../types/Request";
import User from "../../models/User";
import { TypedRequestBody } from "../../types";

const router = Router();

// @route   GET api/auth
// @desc    Get authenticated user given the token
// @access  Private
router.get("/", auth, async (req: RequestWithUserId, res: Response) => {
  try {
    const user = await User.findById(req.userId).select("-password");
    res.json(user);
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

interface AuthPostBody {
  email: string;
  password: string;
}

// @route   POST api/auth
// @desc    Login user and get token
// @access  Public
router.post("/", async (req: TypedRequestBody<AuthPostBody>, res: Response) => {
  const { email, password } = req.body;
  try {
    let user = await User.findOne({ email });

    if (!user) {
      return res.status(HttpStatusCodes.BAD_REQUEST).json({
        errors: [
          {
            msg: "Invalid Credentials",
          },
        ],
      });
    }

    const isMatch = await bcrypt.compare(password, user.password);

    if (!isMatch) {
      return res.status(HttpStatusCodes.BAD_REQUEST).json({
        errors: [
          {
            msg: "Invalid Credentials",
          },
        ],
      });
    }

    const payload = {
      userId: user.id,
    };

    jwt.sign(
      payload,
      process.env.JWT_SECRET ?? "",
      { expiresIn: process.env.JWT_EXPIRATION },
      (err, token) => {
        if (err) throw err;
        res.json({ token });
      }
    );
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

export default router;
