// Applications of analytic combinatorics to algorithms

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

// Routines of general purpose

typedef int bool;
#define true  1
#define false 0

#define trunc(x) ((unsigned int) floor(x))
#define round(x) (trunc((x) + 0.5))

double Calc_Polynom(unsigned int * coeffs, unsigned int arr_Len, double arg) {
  int k;
  double result = 0.0;
  for (k = arr_Len - 1; k >= 0; --k)
    result = result * arg + coeffs[k];
  return result;
}

void printIntArray(unsigned int * arr, unsigned int arr_Len) {
  unsigned int k;
  if (!arr) return;
  if (arr_Len == 0) return;
  printf("\t%u", arr[0]);
  for (k = 1; k < arr_Len; ++k)
    printf("\t%u", arr[k]);
  printf("\n");
}

// Generating functions: types and definition

typedef bool GF_kind;  // Generating functions:
#define OGF true       // - ordinary generating functions;
#define EGF false      // - exponential generating functions.

typedef int GF_def_kind; // Definitions of generating functions.
#define POINTER        0
#define SERIES         1
#define EPSILON        2
#define IDENTITY       3
#define DIFFERENCE     4
#define UNION          5
#define PRODUCT        6
#define POWER          7
#define SEQUENCE       8
#define FAMILY         9
#define CYCLE_OF_LEN  10
#define CYCLE         11

typedef struct {
  GF_def_kind def_kind_GF;
  unsigned int data_Len;
  void * data;
} GeneratingFunction;

// Calculating values of generating functions

double Calc_GF(GeneratingFunction * f, double arg);

double Calc_Pointer(GeneratingFunction * f, double arg) {
  return Calc_GF(f, arg);
}

double Minus(GeneratingFunction * * f, double arg) {
  return Calc_GF(f[0], arg) - Calc_GF(f[1], arg);
}

double Sum(GeneratingFunction * * arr, unsigned int arr_Len, double arg) {
  double result = 0.0;
  unsigned int k;
  for (k = 0; k < arr_Len; ++k)
    result += Calc_GF(arr[k], arg);
  return result;
}

double Product(GeneratingFunction * * arr, unsigned int arr_Len, double arg) {
  double result = 1.0;
  unsigned int k;
  for (k = 0; k < arr_Len; ++k)
    result *= Calc_GF(arr[k], arg);
  return result;
}

double Power(GeneratingFunction * f, unsigned int pow, double arg) {
  double result = 1.0;
  double value = Calc_GF(f, arg);
  unsigned int k;
  for (k = 0; k < pow; ++k)
    result *= value;
  return result;
}

double Calc_GF(GeneratingFunction * f, double arg) {
  switch (f->def_kind_GF) {
    case POINTER      :  return Calc_Pointer(*(GeneratingFunction * *)(f->data), arg);
    case SERIES       :  return Calc_Polynom(f->data, f->data_Len, arg);
    case EPSILON      :  return 1;
    case IDENTITY     :  return arg;
    case DIFFERENCE   :  return Minus(f->data, arg);
    case UNION        :  return Sum(f->data, f->data_Len, arg);
    case PRODUCT      :  return Product(f->data, f->data_Len, arg);
    case POWER        :  return Power(f->data, f->data_Len, arg);
    case SEQUENCE     :  return 1/(1-Calc_GF(f->data, arg));
    case FAMILY       :  return exp(Calc_GF(f->data, arg));
    case CYCLE_OF_LEN :  return Power(f->data, f->data_Len, arg) / f->data_Len;
    case CYCLE        :  return -log(1-Calc_GF(f->data, arg));
  }
  return 0.0; // This command is never executed.
  // It just suppresses a warning for a missing return value.
}

// Constructors of generating functions

#define malloc_GF (GeneratingFunction *) malloc(sizeof(GeneratingFunction))
#define malloc_GF_arr(arr_Len) (GeneratingFunction * *) malloc(arr_Len * sizeof(GeneratingFunction *))

GeneratingFunction * PTR(GeneratingFunction * * g) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = POINTER;
  f->data = g;
  return f;
}

GeneratingFunction * EXPLICIT(unsigned int * coeffs, unsigned int arr_Len) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = SERIES;
  f->data = coeffs;
  f->data_Len = arr_Len;
  return f;
}

GeneratingFunction * EPS(void) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = EPSILON;
  return f;
}

GeneratingFunction * IDENT(void) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = IDENTITY;
  return f;
}

GeneratingFunction * MINUS(GeneratingFunction * g, GeneratingFunction * h) {
  #define arr_Len 2
  GeneratingFunction * * arr = (GeneratingFunction * *) malloc(arr_Len * sizeof(GeneratingFunction *));
  GeneratingFunction * f = malloc_GF;
  arr[0] = g;
  arr[1] = h;
  f->def_kind_GF = DIFFERENCE;
  f->data = arr;
  f->data_Len = arr_Len;
  #undef arr_Len
  return f;
}

GeneratingFunction * SUM(GeneratingFunction * * arr, unsigned int arr_Len) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = UNION;
  f->data = arr;
  f->data_Len = arr_Len;
  return f;
}

GeneratingFunction * SUM_2(
  GeneratingFunction * f,
  GeneratingFunction * g
) {
  #define arr_Len 2
  GeneratingFunction * * arr = malloc_GF_arr(arr_Len);
  arr[0] = f;
  arr[1] = g;
  return SUM(arr, arr_Len);
  #undef arr_Len
}

GeneratingFunction * SUM_3(
  GeneratingFunction * f,
  GeneratingFunction * g,
  GeneratingFunction * h
) {
  #define arr_Len 3
  GeneratingFunction * * arr = malloc_GF_arr(arr_Len);
  arr[0] = f;
  arr[1] = g;
  arr[2] = h;
  return SUM(arr, arr_Len);
  #undef arr_Len
}

GeneratingFunction * PROD(GeneratingFunction * * arr, unsigned int arr_Len) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = PRODUCT;
  f->data = arr;
  f->data_Len = arr_Len;
  return f;
}

GeneratingFunction * PROD_2(
  GeneratingFunction * f,
  GeneratingFunction * g
) {
  #define arr_Len 2
  GeneratingFunction * * arr = malloc_GF_arr(arr_Len);
  arr[0] = f;
  arr[1] = g;
  return PROD(arr, arr_Len);
  #undef arr_Len
}

GeneratingFunction * PROD_3(
  GeneratingFunction * f,
  GeneratingFunction * g,
  GeneratingFunction * h
) {
  #define arr_Len 3
  GeneratingFunction * * arr = malloc_GF_arr(arr_Len);
  arr[0] = f;
  arr[1] = g;
  arr[2] = h;
  return PROD(arr, arr_Len);
  #undef arr_Len
}

GeneratingFunction * POW(GeneratingFunction * g, unsigned int pow) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = POWER;
  f->data = g;
  f->data_Len = pow;
  return f;
}

GeneratingFunction * SEQ(GeneratingFunction * g) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = SEQUENCE;
  f->data = g;
  return f;
}

GeneratingFunction * SET(GeneratingFunction * g) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = FAMILY;
  f->data = g;
  return f;
}

GeneratingFunction * CYC_OF_LEN(GeneratingFunction * g, unsigned int cycle_Len) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = CYCLE_OF_LEN;
  f->data = g;
  f->data_Len = cycle_Len;
  return f;
}

GeneratingFunction * CYC(GeneratingFunction * g) {
  GeneratingFunction * f = malloc_GF;
  f->def_kind_GF = CYCLE;
  f->data = g;
  return f;
}

// Destructors of generating functions

void free_GF(GeneratingFunction * f);

void free_GF_arr(GeneratingFunction * * arr, unsigned int arr_Len) {
  if (arr) {
    unsigned int k;
    for (k = 0; k < arr_Len; ++k)
      free_GF(arr[k]);
    free(arr);
  }
}

void free_GF(GeneratingFunction * f) {
  // Cascade destruction is applied except for "pointed" objects:
  // several "pointers" can point to the same generating function.
  if (!f) return;
  if (f->data) {
    switch (f->def_kind_GF) {
      case POWER        :
      case SEQUENCE     :
      case FAMILY       :
      case CYCLE_OF_LEN :
      case CYCLE        :  free_GF(f->data); break;

      case DIFFERENCE   :
      case UNION        :
      case PRODUCT      :  free_GF_arr(f->data, f->data_Len); break;

      case SERIES       :  free(f->data);
    }
  }
  free(f);
}

unsigned int * findExpansion(
  GeneratingFunction * f,
  unsigned int n,
  GF_kind kind_GF,
  double precision
) {
  // Calculates the first n coefficients of a generating function,
  // multiplied by factorials if kind_GF = EGF.
  // Coefficients are supposed to be non-negative integers!
  // Precision must be a positive real number.

  #define Calc_Coeff                                \
    if (kind_GF) {                                  \
      factorial *= r;                               \
      coeff = round(temp2[n] / factorial);          \
    }                                               \
    else {                                          \
      coeff = round(temp2[n]);                      \
    }                                               \
    result[r] = coeff;

  #define Advance                                   \
    ++r;                                            \
    if (r >= n) break;                              \
    ++left;                                         \
    --right;

  if (!f)
    return NULL;
  else {
    unsigned int * result = (unsigned int *) malloc(n * sizeof(unsigned int));
    unsigned int len = (n << 1) + 1;
    double * temp1 = (double *) malloc(len * sizeof(double));
    double * temp2 = (double *) malloc(len * sizeof(double));
    unsigned int k, r = 0, left = 0, right = len;
    double factorial = 1;
    double sq_precision = precision * precision;
    unsigned int coeff;
    for (k = left; k < right; ++k) {
      double arg = (((int)k) - ((int)n)) * precision;
      temp1[k] = Calc_GF(f, arg);
    }
    precision = 2 * precision;
    result[0] = round(temp1[n]);
    for (;;) {
      Advance
      for (k = left; k < right; ++k)
        temp2[k] = (temp1[k+1] - temp1[k-1]) / precision;
      Calc_Coeff
      Advance
      for (k = left; k < right; ++k)
        temp2[k] = (temp1[k+1] + temp1[k-1] - 2 * temp1[k]) / sq_precision;
      Calc_Coeff
      for (k = left; k < right; ++k)
        temp1[k] = temp2[k];
    }
    free(temp1);
    free(temp2);
    return result;
  }
  #undef Calc_Coeff
  #undef Advance
}

void print_GF(
  GeneratingFunction * f,
  unsigned int n,
  GF_kind kind_GF,
  double precision
) {
  unsigned int * coeffs = findExpansion(f, n, kind_GF, precision);
  if (coeffs) {
    printIntArray(coeffs, n);
    free(coeffs);
  }
  else {
    printf("undefined generating function.\n");
  }
}

// Examples

#define coeffs_Len 6

GeneratingFunction * PositiveIntegers_GF(void) {
  unsigned int * coeffs = (unsigned int *) malloc(coeffs_Len * sizeof(unsigned int));
  unsigned int k;
  coeffs[0] = 0;
  for(k = 1; k < coeffs_Len; ++k)
    coeffs[k] = 1;
  return EXPLICIT(coeffs, coeffs_Len);
}

GeneratingFunction * ZERO(void) {
  unsigned int * coeffs = (unsigned int *) malloc(coeffs_Len * sizeof(unsigned int));
  unsigned int k;
  for(k = 0; k < coeffs_Len; ++k)
    coeffs[k] = 0;
  return EXPLICIT(coeffs, coeffs_Len);
}

#define default_precision 0.001

void print_free_GF(
  GeneratingFunction * f,
  GF_kind kind_GF,
  char * name
) {
  printf("\n");
  printf(name);
  printf(":\n");
  print_GF(f, coeffs_Len, kind_GF, default_precision);
  printf("\n");
  free_GF(f);
}

#undef coeffs_Len

unsigned int * solve_func_equation_by_iteration(
  GeneratingFunction * recursive_GF, // SERIES; already initialized
  GeneratingFunction * f,
  GF_kind kind_GF
) {
  unsigned int k;
  bool equal;
  unsigned int n = recursive_GF->data_Len;
  unsigned int * coeffs1 = (unsigned int *) recursive_GF->data;
  unsigned int * coeffs2;
  beginning:
  coeffs2 = findExpansion(f, n, kind_GF, default_precision);
  equal = true;
  for (k = 0; k < n; ++k) {
    if (coeffs2[k] != coeffs1[k]) {
      equal = false;
      break;
    }
  }
  if (equal) return coeffs2;
  for (k = 0; k < n; ++k)
    coeffs1[k] = coeffs2[k];
  free(coeffs2);
  goto beginning;
}

#undef default_precision

void print_free_rec_GF(
  GeneratingFunction * * recursive_GF_ptr,
  GeneratingFunction * f, // f = equation(recursive_FG)
  GF_kind kind_GF,
  char * name
) {
  GeneratingFunction * recursive_GF = ZERO();
  unsigned int n = recursive_GF->data_Len;
  unsigned int * coeffs;
  * recursive_GF_ptr = recursive_GF;
  coeffs = solve_func_equation_by_iteration(recursive_GF, f, kind_GF);
  free_GF(f);
  free_GF(recursive_GF);
  printf("\n");
  printf(name);
  printf(":\n");
  printIntArray(coeffs, n);
  free(coeffs);
  printf("\n");
}

void non_recursive_examples(void) {
  print_free_GF(
    SEQ(SUM_2(IDENT(), IDENT())),
    OGF,
    "Binary strings"
  );
  print_free_GF(
    SEQ(PositiveIntegers_GF()),
    OGF,
    "Integer compositions" // y = x_1 + x_2 + ... + x_n, x_i > 0
  );
  print_free_GF(
    SET(CYC(IDENT())),
    EGF,
    "Permutations"
  );
  print_free_GF(
    SET(MINUS(CYC(IDENT()), CYC_OF_LEN(IDENT(), 1))),
    EGF,
    "Derangements"
  );
  print_free_GF(
    SET(SUM_2(CYC_OF_LEN(IDENT(), 1), CYC_OF_LEN(IDENT(), 2))),
    EGF,
    "Involutions (telephone numbers)"
  );
}

void recursive_examples(void) {
  GeneratingFunction * Rec_GF; // common place
  print_free_rec_GF(
    &Rec_GF,
    SUM_2(
      EPS(),
      PROD_3(IDENT(), PTR(&Rec_GF), PTR(&Rec_GF))
    ),
    OGF,
    "Ordered binary trees, counting internal nodes (Catalan numbers)"
  );
  print_free_rec_GF(
    &Rec_GF,
    SUM_3(
      IDENT(),
      PROD_2(IDENT(), PTR(&Rec_GF)),
      PROD_3(IDENT(), PTR(&Rec_GF), PTR(&Rec_GF))
    ),
    OGF,
    "Ordered unary-binary trees, counting all nodes"
  );
  print_free_rec_GF(
    &Rec_GF,
    PROD_2(IDENT(), SEQ(PTR(&Rec_GF))),
    OGF,
    "General ordered trees, counting all nodes (Catalan numbers shifted right)"
  );
}

void examples(void) {
  non_recursive_examples();
  recursive_examples();
}

// Main

void main(void) {
  examples();
}