"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const http_status_codes_1 = __importDefault(require("http-status-codes"));
const auth_1 = require("../../middleware/auth");
const Profile_1 = __importDefault(require("../../models/Profile"));
const User_1 = __importDefault(require("../../models/User"));
const router = express_1.Router();
// @route   GET api/profile/me
// @desc    Get current user's profile
// @access  Private
router.get("/me", auth_1.auth, (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const profile = yield Profile_1.default.findOne({
            user: req.userId,
        }).populate("user", ["email"]);
        if (!profile) {
            return res.status(http_status_codes_1.default.BAD_REQUEST).json({
                errors: [
                    {
                        msg: "There is no profile for this user",
                    },
                ],
            });
        }
        res.json(profile);
    }
    catch (err) {
        console.error(err.message);
        res.status(http_status_codes_1.default.INTERNAL_SERVER_ERROR).send("Server Error");
    }
}));
// @route   POST api/profile
// @desc    Create or update user's profile
// @access  Private
router.post("/", auth_1.auth, (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    // validate
    const { firstName, lastName, username } = req.body;
    // Build profile object based on IProfile
    const profileFields = {
        user: req.userId,
        firstName,
        lastName,
        username,
    };
    try {
        let user = yield User_1.default.findOne({ _id: req.userId });
        if (!user) {
            return res.status(http_status_codes_1.default.BAD_REQUEST).json({
                errors: [
                    {
                        msg: "User not registered",
                    },
                ],
            });
        }
        let profile = yield Profile_1.default.findOne({ user: req.userId });
        if (profile) {
            // Update
            profile = yield Profile_1.default.findOneAndUpdate({ user: req.userId }, { $set: profileFields }, { new: true });
            return res.json(profile);
        }
        // Create
        profile = new Profile_1.default(profileFields);
        yield profile.save();
        res.json(profile);
    }
    catch (err) {
        console.error(err.message);
        res.status(http_status_codes_1.default.INTERNAL_SERVER_ERROR).send("Server Error");
    }
}));
// @route   GET api/profile
// @desc    Get all profiles
// @access  Public
router.get("/", (_, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const profiles = yield Profile_1.default.find().populate("user", ["email"]);
        res.json(profiles);
    }
    catch (err) {
        console.error(err.message);
        res.status(http_status_codes_1.default.INTERNAL_SERVER_ERROR).send("Server Error");
    }
}));
// @route   GET api/profile/user/:userId
// @desc    Get profile by userId
// @access  Public
router.get("/user/:userId", (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const profile = yield Profile_1.default.findOne({
            user: req.params.userId,
        }).populate("user", ["email"]);
        if (!profile) {
            return res
                .status(http_status_codes_1.default.NOT_FOUND)
                .json({ msg: "Profile not found" });
        }
        res.json(profile);
    }
    catch (err) {
        console.error(err.message);
        if (err.kind === "ObjectId") {
            return res
                .status(http_status_codes_1.default.NOT_FOUND)
                .json({ msg: "Profile not found" });
        }
        res.status(http_status_codes_1.default.INTERNAL_SERVER_ERROR).send("Server Error");
    }
}));
// @route   DELETE api/profile
// @desc    Delete profile and user
// @access  Private
router.delete("/", auth_1.auth, (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Remove profile
        yield Profile_1.default.findOneAndRemove({ user: req.userId });
        // Remove user
        yield User_1.default.findOneAndRemove({ _id: req.userId });
        res.json({ msg: "User removed" });
    }
    catch (err) {
        console.error(err.message);
        res.status(http_status_codes_1.default.INTERNAL_SERVER_ERROR).send("Server Error");
    }
}));
exports.default = router;
//# sourceMappingURL=profile.js.map