import { Router, Response } from "express";
import HttpStatusCodes from "http-status-codes";
import Book, { IBook } from "../../models/Book";
import {
  TypedRequestBody,
  TypedRequestQuery,
  TypedResponse,
} from "../../types";
import { RequestWithUserId } from "../../types/Request";

const router = Router();

type GetAllBooksQuery = {
  page?: string;
  pageSize?: string;
};

type GetAllBooksResponse = {
  books: IBook[];
  nextPage: number | null;
  total: number;
};

type PostBookBody = {
  title: string;
  authorName: string;
};

// @route   GET api/books
// @desc    Get all books, pagination is enabled
// @access  Public
router.get(
  "/",
  async (
    req: TypedRequestQuery<GetAllBooksQuery>,
    res: TypedResponse<GetAllBooksResponse>
  ) => {
    try {
      const { page = "1", pageSize = "10" } = req.query;
      const pageNumber = Number(page);
      const pageSizeNumber = Number(pageSize);
      const books = await Book.find()
        .skip((pageNumber - 1) * pageSizeNumber)
        .limit(pageSizeNumber);
      const total: number = await Book.countDocuments();
      res.json({
        books,
        nextPage: pageNumber * pageSizeNumber < total ? pageNumber + 1 : null,
        total: Math.ceil(total / pageSizeNumber),
      });
    } catch (err) {
      console.error((err as any).message);
      res.status(HttpStatusCodes.NOT_FOUND).send("Server Error");
    }
  }
);

// @route   POST api/books
// @desc    Create or update books
// @access  Public
router.post("/", async (req: TypedRequestBody<PostBookBody>, res: Response) => {
  // validate
  const { authorName, title } = req.body;

  const bookFields = {
    authorName,
    title,
  };

  try {
    const book = new Book(bookFields);
    await book.save();
    res.json(book);
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

// @route   GET api/books/:bookId
// @desc    Get book by bookId
// @access  Public
router.get("/:bookId", async (req: RequestWithUserId, res: Response) => {
  try {
    const book = await Book.findById(req.params.bookId);
    res.json(book);
  } catch (err) {
    console.error((err as any).message);
    if ((err as any).kind === "ObjectId") {
      return res
        .status(HttpStatusCodes.NOT_FOUND)
        .json({ msg: "Book not found" });
    }
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

// @route   DELETE api/books/:bookId
// @desc    Delete books
// @access  Public
router.delete("/:bookId", async (req: RequestWithUserId, res: Response) => {
  try {
    await Book.findByIdAndRemove(req.params.bookId);

    res.json({ msg: "Book removed." });
  } catch (err) {
    console.error((err as any).message);
    res.status(HttpStatusCodes.INTERNAL_SERVER_ERROR).send("Server Error");
  }
});

export default router;
