#include "Polynomial.h"

typedef LinkedListIterator<PolynomialMember> PolynomialIterator;

//PolynomialMember implementation
void PolynomialMember::operator+=(const PolynomialMember& member) {
	this->coefficient += member.coefficient;
}

void PolynomialMember::operator*=(const PolynomialMember& member) {
	coefficient *= member.coefficient;
	power += member.power;
}

float PolynomialMember::operator()(float x) {
	return pow(x,power)*coefficient;
}

//default constructor constructs the member x
PolynomialMember::PolynomialMember() {
	this->power = 1;
	this->coefficient = 1;
}

PolynomialMember::PolynomialMember(float coefficient, int power) {
	this->power = power;
	this->coefficient = coefficient;
}

//Polynomial implementation
Polynomial& Polynomial::addMember(const PolynomialMember& pMember) {
	if (pMember.coefficient == 0) return *this;
	PolynomialIterator iterator = members.begin();
	bool added = false;
	while (iterator && !added) {
		if ((*iterator).power == pMember.power) {
			(*iterator)+=pMember;
			added = true;
		}
		//sorted insert
		if ((*iterator).power < pMember.power) {
			members.insertBefore(pMember, iterator);
			added = true;
		}

		if ((*iterator).coefficient == 0) {
			iterator++;
			PolynomialMember bulk;
			if (iterator) {
				members.deleteBefore(bulk, iterator);
			} else {
				//we've reached the end
				members.deleteEnd(bulk);
			}
		} else {
			iterator++;
		}
	}

	//the new member has the lowest power in the polynomial
	if (!added) {
		members.insertEnd(pMember);
	}

	return *this;
}

float Polynomial::operator()(const float x) {
	PolynomialIterator iterator = members.begin();
	float result = 0.0;
	
	while (iterator) {
		result += (*iterator)(x);
		iterator++;
	}

	return result;
}

void Polynomial::print() {
	PolynomialIterator iterator = members.begin();
	bool isFirstMember = true;
	
	while (iterator) {
		//we show '+' when the coefficient is positive and the member is not the first one
		if (((*iterator).coefficient > 1) && !isFirstMember) 
			cout<<"+";
	
		//we omit the coefficient when it has a value of 1
		if ((*iterator).coefficient != 1)
			cout<<(*iterator).coefficient;
		else
			if (!isFirstMember)
				cout<<"+";

		//we show the power only when its value is above 1
		if ((*iterator).power != 0) {
			if ((*iterator).coefficient != 1) cout<<"*";
			cout<<"x";
			if ((*iterator).power > 1) {
				cout<<"^"<<(*iterator).power;
			}
		} else {
			//if the member's power is 0 we must show 1 only when the coefficient is 1 - otherwise we'll print out things like -6*1
			if ((*iterator).coefficient == 1) {
				cout<<"1";
			}
		}
		
		iterator++;
		isFirstMember = false;
	}
}

void Polynomial::add(const Polynomial& otherPolynomial) {
	PolynomialIterator iterator = otherPolynomial.members.begin();
	
	while (iterator) {
		addMember(*iterator);
		iterator++;
	}
}

void Polynomial::memberMultiply(const PolynomialMember& member) {
	PolynomialIterator iterator = members.begin();
	
	while (iterator) {
		(*iterator) *= member;
		iterator++;
	}
}

void Polynomial::polynomialMultiply(const Polynomial& otherPolynomial) {
	PolynomialIterator iterator = otherPolynomial.members.begin();
	Polynomial result;

	while (iterator) {
		Polynomial addend(*this);

		addend.memberMultiply(*iterator);
		result.add(addend);

		iterator++;
	}

	members = result.members;
}

//Denominator -> , Denominator -> , Quotient -> , Reminder -> 
PolynomialMember getQuotientAddend(LinkedList<PolynomialMember> denominatorMembers, LinkedList<PolynomialMember> quotientMembers) {
	int addendPower = -1;
	float addendCoefficient = 0.0f;
	
	if (!denominatorMembers.empty() && !quotientMembers.empty()) {
		addendPower = (*denominatorMembers.begin()).power - (*quotientMembers.begin()).power;
		addendCoefficient = (*denominatorMembers.begin()).coefficient / (*quotientMembers.begin()).coefficient;
	}

	return PolynomialMember(addendCoefficient, addendPower);
}

void Polynomial::divide(const Polynomial denominator, Polynomial& quotient, Polynomial& reminder) {
	PolynomialMember currentAddend = getQuotientAddend(members, denominator.members),
		negativeOneMember = PolynomialMember(-1, 0);
	reminder = *this;
	
	while (currentAddend.power >= 0) {
		quotient.addMember(currentAddend);

		Polynomial tempP(denominator);
		tempP.memberMultiply(currentAddend);
		tempP.memberMultiply(negativeOneMember);
		reminder.add(tempP);

		currentAddend = getQuotientAddend(reminder.members, denominator.members);
	}
}
